<?php
require_once '../config.php';
require_once '../includes/user_activity_logger.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

// Get current user info
$current_user_id = $_SESSION['user_id'];
$current_user_role = $_SESSION['role'];

// Determine which user's logs to show
$view_user_id = $current_user_id; // Default to current user
$view_username = $_SESSION['username'];

// Super admin and admin can view other users' logs
if (($current_user_role === 'super_admin' || $current_user_role === 'admin') && isset($_GET['user_id'])) {
    $requested_user_id = (int)$_GET['user_id'];

    // Verify the requested user exists
    $stmt = $pdo->prepare("SELECT id, username FROM users WHERE id = ?");
    $stmt->execute([$requested_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($user_data) {
        $view_user_id = $requested_user_id;
        $view_username = $user_data['username'];
    }
}

// Handle filters
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$action_filter = $_GET['action'] ?? '';
$limit = (int)($_GET['limit'] ?? 50);
$offset = (int)($_GET['offset'] ?? 0);

// Get user logs
$user_logs = UserActivityLogger::getUserLogs($view_user_id, $limit, $offset, $date_from, $date_to, $action_filter);

// Get activity statistics
$activity_stats = UserActivityLogger::getActivityStats($view_user_id);

// Get all users for admin and super admin dropdown
$all_users = [];
if ($current_user_role === 'super_admin' || $current_user_role === 'admin') {
    $stmt = $pdo->query("SELECT id, username, role FROM users ORDER BY username");
    $all_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

include '../includes/admin_header.php';
?>

<div class="user-logs-container">
    <div class="page-header">
        <h2>📋 User Activity Logs</h2>
        <p>Monitor and track user activities in the system</p>
    </div>

    <!-- User Selection for Admin and Super Admin -->
    <?php if ($current_user_role === 'super_admin' || $current_user_role === 'admin'): ?>
        <div class="user-selector-card">
            <div class="user-selector-header">
                <h3>Select User to View Logs</h3>
            </div>
            <form method="GET" class="user-selector-form">
                <div class="form-group">
                    <label for="user_id">Select User:</label>
                    <select id="user_id" name="user_id" onchange="this.form.submit()">
                        <option value="">-- Select User --</option>
                        <?php foreach ($all_users as $user): ?>
                            <option value="<?php echo $user['id']; ?>"
                                    <?php echo ($view_user_id == $user['id']) ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($user['username']); ?> (<?php echo ucwords(str_replace('_', ' ', $user['role'])); ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </form>
        </div>
    <?php endif; ?>

    <!-- Current User Info -->
    <div class="current-user-info">
        <div class="info-card">
            <h4>Viewing Logs for: <strong><?php echo htmlspecialchars($view_username); ?></strong></h4>
            <p>User ID: <?php echo $view_user_id; ?> | Role: <?php echo ucwords(str_replace('_', ' ', $_SESSION['role'])); ?></p>
        </div>
    </div>

    <!-- Activity Statistics -->
    <?php if (!empty($activity_stats)): ?>
        <div class="stats-card">
            <div class="stats-header">
                <h3>📊 Activity Summary (Last 30 Days)</h3>
            </div>
            <div class="stats-grid">
                <?php foreach ($activity_stats as $stat): ?>
                    <div class="stat-item">
                        <div class="stat-icon">
                            <?php
                            $icon = 'fas fa-cog';
                            switch ($stat['action']) {
                                case 'login': $icon = 'fas fa-sign-in-alt'; break;
                                case 'logout': $icon = 'fas fa-sign-out-alt'; break;
                                case 'create_user': $icon = 'fas fa-user-plus'; break;
                                case 'update_user': $icon = 'fas fa-user-edit'; break;
                                case 'delete_user': $icon = 'fas fa-user-times'; break;
                                case 'view_page': $icon = 'fas fa-eye'; break;
                                case 'add_grades': $icon = 'fas fa-graduation-cap'; break;
                                case 'update_grades': $icon = 'fas fa-edit'; break;
                            }
                            ?>
                            <i class="<?php echo $icon; ?>"></i>
                        </div>
                        <div class="stat-content">
                            <h4><?php echo ucwords(str_replace('_', ' ', $stat['action'])); ?></h4>
                            <p><?php echo $stat['count']; ?> times</p>
                            <small>Last: <?php echo date('M d, H:i', strtotime($stat['last_activity'])); ?></small>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    <?php endif; ?>

    <!-- Filters -->
    <div class="filters-card">
        <div class="filters-header">
            <h3>🔍 Filter Logs</h3>
        </div>
        <form method="GET" class="filters-form">
            <?php if (($current_user_role === 'super_admin' || $current_user_role === 'admin') && isset($_GET['user_id'])): ?>
                <input type="hidden" name="user_id" value="<?php echo $view_user_id; ?>">
            <?php endif; ?>

            <div class="form-row">
                <div class="form-group">
                    <label for="date_from">From Date:</label>
                    <input type="date" id="date_from" name="date_from" value="<?php echo $date_from; ?>">
                </div>

                <div class="form-group">
                    <label for="date_to">To Date:</label>
                    <input type="date" id="date_to" name="date_to" value="<?php echo $date_to; ?>">
                </div>

                <div class="form-group">
                    <label for="action">Action:</label>
                    <select id="action" name="action">
                        <option value="">All Actions</option>
                        <option value="login" <?php echo ($action_filter === 'login') ? 'selected' : ''; ?>>Login</option>
                        <option value="logout" <?php echo ($action_filter === 'logout') ? 'selected' : ''; ?>>Logout</option>
                        <option value="create_user" <?php echo ($action_filter === 'create_user') ? 'selected' : ''; ?>>Create User</option>
                        <option value="update_user" <?php echo ($action_filter === 'update_user') ? 'selected' : ''; ?>>Update User</option>
                        <option value="delete_user" <?php echo ($action_filter === 'delete_user') ? 'selected' : ''; ?>>Delete User</option>
                        <option value="view_page" <?php echo ($action_filter === 'view_page') ? 'selected' : ''; ?>>View Page</option>
                        <option value="add_grades" <?php echo ($action_filter === 'add_grades') ? 'selected' : ''; ?>>Add Grades</option>
                        <option value="update_grades" <?php echo ($action_filter === 'update_grades') ? 'selected' : ''; ?>>Update Grades</option>
                    </select>
                </div>

                <div class="form-group">
                    <label for="limit">Show:</label>
                    <select id="limit" name="limit" onchange="this.form.submit()">
                        <option value="25" <?php echo ($limit == 25) ? 'selected' : ''; ?>>25 entries</option>
                        <option value="50" <?php echo ($limit == 50) ? 'selected' : ''; ?>>50 entries</option>
                        <option value="100" <?php echo ($limit == 100) ? 'selected' : ''; ?>>100 entries</option>
                    </select>
                </div>
            </div>

            <div class="filter-actions">
                <button type="submit" class="btn-user">Apply Filters</button>
                <a href="?<?php echo ($current_user_role === 'super_admin' || $current_user_role === 'admin') && isset($_GET['user_id']) ? 'user_id=' . $view_user_id : ''; ?>" class="btn-user btn-warning">Clear Filters</a>
            </div>
        </form>
    </div>

    <!-- Activity Logs Table -->
    <div class="logs-card">
        <div class="logs-header">
            <h3>📝 Activity Logs</h3>
            <div class="logs-count">
                Showing <?php echo count($user_logs); ?> entries
            </div>
        </div>

        <?php if (empty($user_logs)): ?>
            <div class="no-logs">
                <i class="fas fa-inbox"></i>
                <h4>No activity logs found</h4>
                <p>No activities have been recorded for this user yet.</p>
            </div>
        <?php else: ?>
            <div class="table-container">
                <table class="logs-table">
                    <thead>
                        <tr>
                            <th>Date & Time</th>
                            <th>Action</th>
                            <th>Description</th>
                            <th>IP Address</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($user_logs as $log): ?>
                            <tr>
                                <td>
                                    <div class="datetime">
                                        <strong><?php echo date('M d, Y', strtotime($log['created_at'])); ?></strong><br>
                                        <small><?php echo date('H:i:s', strtotime($log['created_at'])); ?></small>
                                    </div>
                                </td>
                                <td>
                                    <span class="action-badge action-<?php echo $log['action']; ?>">
                                        <?php
                                        $action_icon = 'fas fa-cog';
                                        switch ($log['action']) {
                                            case 'login': $action_icon = 'fas fa-sign-in-alt'; break;
                                            case 'logout': $action_icon = 'fas fa-sign-out-alt'; break;
                                            case 'create_user': $action_icon = 'fas fa-user-plus'; break;
                                            case 'update_user': $action_icon = 'fas fa-user-edit'; break;
                                            case 'delete_user': $action_icon = 'fas fa-user-times'; break;
                                            case 'view_page': $action_icon = 'fas fa-eye'; break;
                                            case 'add_grades': $action_icon = 'fas fa-graduation-cap'; break;
                                            case 'update_grades': $action_icon = 'fas fa-edit'; break;
                                        }
                                        ?>
                                        <i class="<?php echo $action_icon; ?>"></i>
                                        <?php echo ucwords(str_replace('_', ' ', $log['action'])); ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($log['description']); ?></td>
                                <td>
                                    <code><?php echo htmlspecialchars($log['ip_address']); ?></code>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <?php if ($offset > 0 || count($user_logs) >= $limit): ?>
                <div class="pagination">
                    <?php if ($offset > 0): ?>
                        <a href="?<?php echo http_build_query(array_merge($_GET, ['offset' => max(0, $offset - $limit)])); ?>" class="btn-user btn-info">
                            <i class="fas fa-chevron-left"></i> Previous
                        </a>
                    <?php endif; ?>

                    <?php if (count($user_logs) >= $limit): ?>
                        <a href="?<?php echo http_build_query(array_merge($_GET, ['offset' => $offset + $limit])); ?>" class="btn-user btn-info">
                            Next <i class="fas fa-chevron-right"></i>
                        </a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<style>
    .user-logs-container {
        padding: 20px;
    }

    .page-header {
        margin-bottom: 30px;
    }

    .page-header h2 {
        color: #2c3e50;
        margin-bottom: 10px;
        font-size: 2em;
    }

    .page-header p {
        color: #7f8c8d;
        margin: 0;
        font-size: 1.1em;
    }

    .user-selector-card,
    .current-user-info,
    .stats-card,
    .filters-card,
    .logs-card {
        background: white;
        border-radius: 10px;
        padding: 25px;
        margin-bottom: 20px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        border: 1px solid #f0f0f0;
    }

    .user-selector-header,
    .stats-header,
    .filters-header,
    .logs-header {
        margin-bottom: 20px;
        padding-bottom: 15px;
        border-bottom: 2px solid #3498db;
    }

    .user-selector-header h3,
    .stats-header h3,
    .filters-header h3,
    .logs-header h3 {
        margin: 0;
        color: #2c3e50;
        font-size: 1.4em;
    }

    .info-card {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 20px;
        border-radius: 8px;
    }

    .info-card h4 {
        margin: 0 0 10px 0;
        font-size: 1.2em;
    }

    .info-card p {
        margin: 0;
        opacity: 0.9;
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 15px;
    }

    .stat-item {
        background: #f8f9fa;
        padding: 20px;
        border-radius: 8px;
        display: flex;
        align-items: center;
        border: 1px solid #e9ecef;
    }

    .stat-icon {
        font-size: 2em;
        margin-right: 15px;
        color: #3498db;
    }

    .stat-content h4 {
        margin: 0 0 5px 0;
        color: #2c3e50;
        font-size: 1.1em;
    }

    .stat-content p {
        margin: 0 0 5px 0;
        font-size: 1.2em;
        font-weight: bold;
        color: #3498db;
    }

    .stat-content small {
        color: #6c757d;
        font-size: 0.9em;
    }

    .form-row {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 15px;
        margin-bottom: 20px;
    }

    .form-group {
        display: flex;
        flex-direction: column;
    }

    .form-group label {
        margin-bottom: 5px;
        font-weight: 600;
        color: #2c3e50;
    }

    .form-group input,
    .form-group select {
        padding: 10px;
        border: 1px solid #ddd;
        border-radius: 5px;
        font-size: 1em;
    }

    .filter-actions {
        display: flex;
        gap: 15px;
        justify-content: flex-start;
        margin-top: 20px;
        padding-top: 20px;
        border-top: 1px solid #e9ecef;
    }

    .btn-user {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        padding: 12px 24px;
        border: none;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none !important;
        cursor: pointer;
        transition: all 0.3s ease;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
        color: white;
    }

    .btn-user:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0,0,0,0.15);
        background: linear-gradient(135deg, #2980b9 0%, #21618c 100%);
    }

    .btn-user:active {
        transform: translateY(0);
        box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    }

    .btn-user:focus {
        outline: none;
        box-shadow: 0 0 0 3px rgba(52, 152, 219, 0.3);
    }

    .btn-warning {
        background: linear-gradient(135deg, #f39c12 0%, #e67e22 100%);
        color: white;
    }

    .btn-warning:hover {
        background: linear-gradient(135deg, #e67e22 0%, #d35400 100%);
    }

    .btn-info {
        background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);
        color: white;
    }

    .btn-info:hover {
        background: linear-gradient(135deg, #138496 0%, #117a8b 100%);
    }

    .logs-count {
        color: #6c757d;
        font-size: 0.9em;
    }

    .no-logs {
        text-align: center;
        padding: 50px 20px;
        color: #6c757d;
    }

    .no-logs i {
        font-size: 3em;
        margin-bottom: 20px;
        opacity: 0.5;
    }

    .no-logs h4 {
        margin: 0 0 10px 0;
        color: #2c3e50;
    }

    .table-container {
        overflow-x: auto;
    }

    .logs-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .logs-table th,
    .logs-table td {
        padding: 15px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .logs-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #2c3e50;
        border-bottom: 2px solid #3498db;
    }

    .logs-table tr:hover {
        background: #f8f9fa;
    }

    .datetime {
        line-height: 1.4;
    }

    .action-badge {
        display: inline-flex;
        align-items: center;
        gap: 5px;
        padding: 5px 10px;
        border-radius: 15px;
        font-size: 0.85em;
        font-weight: 500;
        text-transform: capitalize;
    }

    .action-login { background-color: #d4edda; color: #155724; }
    .action-logout { background-color: #f8d7da; color: #721c24; }
    .action-create_user { background-color: #d1ecf1; color: #0c5460; }
    .action-update_user { background-color: #fff3cd; color: #856404; }
    .action-delete_user { background-color: #f8d7da; color: #721c24; }
    .action-view_page { background-color: #e2e3e5; color: #383d41; }
    .action-add_grades { background-color: #d4edda; color: #155724; }
    .action-update_grades { background-color: #cce5ff; color: #004085; }

    .pagination {
        display: flex;
        justify-content: center;
        gap: 10px;
        margin-top: 20px;
    }

    @media (max-width: 768px) {
        .user-logs-container {
            padding: 10px;
        }

        .form-row {
            grid-template-columns: 1fr;
        }

        .stats-grid {
            grid-template-columns: 1fr;
        }

        .stat-item {
            flex-direction: column;
            text-align: center;
        }

        .stat-icon {
            margin-right: 0;
            margin-bottom: 10px;
        }

        .filter-actions {
            flex-direction: column;
        }

        .pagination {
            flex-direction: column;
            align-items: center;
        }
    }
</style>

<script>
    // Auto-refresh functionality (optional)
    let autoRefresh = false;

    function toggleAutoRefresh() {
        autoRefresh = !autoRefresh;
        if (autoRefresh) {
            setInterval(() => {
                location.reload();
            }, 30000); // Refresh every 30 seconds
        }
    }

    // Enhanced table interactions
    document.addEventListener('DOMContentLoaded', function() {
        // Add click handlers for expandable log details if needed
        const logRows = document.querySelectorAll('.logs-table tbody tr');
        logRows.forEach(row => {
            row.addEventListener('click', function() {
                // Could expand to show more details
                this.classList.toggle('expanded');
            });
        });
    });
</script>

<?php
include '../includes/admin_footer.php';
?>
