<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../config.php';
require_once '../includes/user_activity_logger.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

// Check if user has permission to manage student users
require_once '../includes/enhanced_access_control_fixed.php';
requirePermission('student_logins');

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'update_student':
                updateStudentUser();
                break;
            case 'deactivate_student':
                deactivateStudentUser();
                break;
            case 'delete_student':
                deleteStudentUser();
                break;
        }
    }
}

// Update student user
function updateStudentUser() {
    global $pdo;

    try {
        $update_fields = ['username = ?'];
        $params = [$_POST['username']];

        // Update password only if provided
        if (!empty($_POST['password'])) {
            // Validate strong password
            if (!validateStrongPassword($_POST['password'])) {
                $_SESSION['error_message'] = 'Password must be at least 8 characters long and contain at least one uppercase letter, one lowercase letter, one number, and one special character.';
                return;
            }
            $update_fields[] = 'password = ?';
            $params[] = password_hash($_POST['password'], PASSWORD_DEFAULT);
        }

        $params[] = $_POST['user_id'];

        $stmt = $pdo->prepare("
            UPDATE student_users
            SET " . implode(', ', $update_fields) . "
            WHERE id = ?
        ");

        $stmt->execute($params);

        // Log student user update activity
        $password_changed = !empty($_POST['password']) ? ' (password changed)' : '';
        $description = "Updated student user ID: {$_POST['user_id']} - Username: {$_POST['username']}{$password_changed}";
        UserActivityLogger::logActivity($_SESSION['user_id'], 'update_student_user', $description);

        $_SESSION['success_message'] = 'Student user updated successfully!';
        header('Location: student_accounts.php');
        exit;
    } catch (PDOException $e) {
        $_SESSION['error_message'] = 'Error updating student user: ' . $e->getMessage();
    }
}

// Deactivate student user
function deactivateStudentUser() {
    global $pdo;

    try {
        $stmt = $pdo->prepare("UPDATE student_users SET status = 'inactive' WHERE id = ?");
        $stmt->execute([$_POST['user_id']]);

        // Log student user deactivation activity
        $description = "Deactivated student user ID: {$_POST['user_id']}";
        UserActivityLogger::logActivity($_SESSION['user_id'], 'deactivate_student_user', $description);

        $_SESSION['success_message'] = 'Student user deactivated successfully!';
        header('Location: student_accounts.php');
        exit;
    } catch (PDOException $e) {
        $_SESSION['error_message'] = 'Error deactivating student user: ' . $e->getMessage();
    }
}

// Delete student user
function deleteStudentUser() {
    global $pdo;

    try {
        // First, delete from students table if exists
        $stmt = $pdo->prepare("DELETE FROM students WHERE user_id = ?");
        $stmt->execute([$_POST['user_id']]);

        // Then delete from student_users table
        $stmt = $pdo->prepare("DELETE FROM student_users WHERE id = ?");
        $stmt->execute([$_POST['user_id']]);

        // Log student user deletion activity
        $description = "Deleted student user ID: {$_POST['user_id']}";
        UserActivityLogger::logActivity($_SESSION['user_id'], 'delete_student_user', $description);

        $_SESSION['success_message'] = 'Student user deleted successfully!';
        header('Location: student_accounts.php');
        exit;
    } catch (PDOException $e) {
        $_SESSION['error_message'] = 'Error deleting student user: ' . $e->getMessage();
    }
}

// Validate strong password
function validateStrongPassword($password) {
    // At least 8 characters, one uppercase, one lowercase, one number, one special character
    return preg_match('/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/', $password);
}

$per_page = 10; // Students per page
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $per_page;
$search_term = $_GET['search'] ?? '';

try {
    // Prepare base count query with optional WHERE clause for search
    $count_query = "SELECT COUNT(*) as total FROM student_users su LEFT JOIN students s ON su.id = s.user_id LEFT JOIN classes cl ON s.class = cl.name";
    $count_params = [];
    if ($search_term !== '') {
        $count_query .= " WHERE su.username LIKE ? OR s.first_name LIKE ? OR s.last_name LIKE ?";
        $like_search = '%' . $search_term . '%';
        $count_params = [$like_search, $like_search, $like_search];
    }
    $count_stmt = $pdo->prepare($count_query);
    $count_stmt->execute($count_params);
    $total_students = $count_stmt->fetch()['total'];
    $total_pages = ceil($total_students / $per_page);

    // Prepare base data query with optional WHERE clause for search
    $data_query = "
        SELECT su.id, su.username, su.created_at, su.status, s.first_name, s.last_name, cl.name as class_name
        FROM student_users su
        LEFT JOIN students s ON su.id = s.user_id
        LEFT JOIN classes cl ON s.class = cl.name
    ";
    $data_params = [];
    if ($search_term !== '') {
        $data_query .= " WHERE su.username LIKE ? OR s.first_name LIKE ? OR s.last_name LIKE ?";
        $like_search = '%' . $search_term . '%';
        $data_params = [$like_search, $like_search, $like_search];
    }
    $data_query .= " ORDER BY s.last_name ASC, s.first_name ASC, su.created_at DESC LIMIT ? OFFSET ?";

    $data_stmt = $pdo->prepare($data_query);
    // Bind data params plus pagination params
    $bind_index = 1;
    foreach ($data_params as $param) {
        $data_stmt->bindValue($bind_index++, $param, PDO::PARAM_STR);
    }
    $data_stmt->bindValue($bind_index++, $per_page, PDO::PARAM_INT);
    $data_stmt->bindValue($bind_index++, $offset, PDO::PARAM_INT);
    $data_stmt->execute();
    $students = $data_stmt->fetchAll();

    // Debug: Check what data we're getting
    if (empty($students)) {
        $_SESSION['error_message'] = 'No student data found. Debug: Total users: ' . $total_students . ', Page: ' . $page . ', Offset: ' . $offset;
    } else {
        // Debug: Log all student data
        foreach ($students as $index => $student) {
            error_log("Debug - Student $index: ID={$student['id']}, Username={$student['username']}, FirstName='{$student['first_name']}', LastName='{$student['last_name']}'");
        }
    }
} catch (PDOException $e) {
    $students = [];
    $total_pages = 1;
    $_SESSION['error_message'] = 'Error fetching student users: ' . $e->getMessage();
}

include '../includes/admin_header.php';
?>

<div class="student-users-container">
    <div class="page-header">
        <h2>👨‍🎓 Student Users Management</h2>
        <p>Manage student user accounts and their login credentials</p>
    </div>

    <!-- Success/Error Messages -->
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="alert alert-success">
            <?php echo $_SESSION['success_message']; unset($_SESSION['success_message']); ?>
        </div>
    <?php endif; ?>

    <?php if (isset($_SESSION['error_message'])): ?>
        <div class="alert alert-error">
            <?php echo $_SESSION['error_message']; unset($_SESSION['error_message']); ?>
        </div>
    <?php endif; ?>

    <!-- Student Users Table -->
    <div class="user-card">
        <div class="user-header">
            <h3 class="user-title">Student User Accounts</h3>
            <div class="table-filters">
                <div class="table-search-container">
        <form method="get" class="table-search-container" action="student_accounts.php">
            <input type="text" name="search" class="table-search" placeholder="Search students..." value="<?php echo htmlspecialchars($search_term); ?>">
            <button type="submit" style="margin-left: 5px; padding: 7px 12px;">Search</button>
            <input type="hidden" name="page" value="1">
        </form>
                </div>
            </div>
        </div>

        <div class="table-container">
            <table class="user-table" id="studentsTable">
                <thead>
                    <tr>
                <th class="sortable">First Name</th>
                <th class="sortable">Last Name</th>
                <th class="sortable">Class</th>
                <th class="sortable">Username</th>
                <th class="sortable">Status</th>
                <th class="sortable">Created Date</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($students as $student): ?>
                <tr>
                    <td><?php echo htmlspecialchars($student['first_name'] ?? ''); ?></td>
                    <td><?php echo htmlspecialchars($student['last_name'] ?? ''); ?></td>
                    <td><?php echo htmlspecialchars($student['class_name'] ?? ''); ?></td>
                    <td><?php echo htmlspecialchars($student['username']); ?></td>
                    <td>
                        <span class="status-badge <?php echo strtolower($student['status'] ?? 'inactive'); ?>">
                            <?php echo ucfirst($student['status'] ?? 'Inactive'); ?>
                        </span>
                    </td>
                    <td><?php echo date('M d, Y', strtotime($student['created_at'])); ?></td>
                    <td>
                        <div class="action-buttons">
                            <button onclick="editStudent(<?php echo htmlspecialchars(json_encode($student)); ?>)"
                                    class="btn-user btn-warning" style="font-size: 0.8em; padding: 5px 10px;">
                                Edit
                            </button>
                            <?php if (($student['status'] ?? 'inactive') === 'active'): ?>
                                <button onclick="deactivateStudent(<?php echo $student['id']; ?>)"
                                        class="btn-user btn-danger" style="font-size: 0.8em; padding: 5px 10px;">
                                    Deactivate
                                </button>
                            <?php endif; ?>
                            <button onclick="deleteStudent(<?php echo $student['id']; ?>)"
                                    class="btn-user btn-danger" style="font-size: 0.8em; padding: 5px 10px;">
                                Delete
                            </button>
                        </div>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
        </div>

        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
            <div class="pagination-container">
                <div class="pagination">
        <?php
        $base_url = '?page=';
        if ($search_term !== '') {
            $base_url = '?search=' . urlencode($search_term) . '&page=';
        }
        ?>

        <?php if ($page > 1): ?>
            <a href="<?php echo $base_url . ($page - 1); ?>" class="page-link">&laquo; Previous</a>
        <?php endif; ?>

        <?php
        $start_page = max(1, $page - 2);
        $end_page = min($total_pages, $page + 2);

        if ($start_page > 1): ?>
            <a href="<?php echo $base_url . '1'; ?>" class="page-link">1</a>
            <?php if ($start_page > 2): ?>
                <span class="page-dots">...</span>
            <?php endif; ?>
        <?php endif; ?>

        <?php for ($i = $start_page; $i <= $end_page; $i++): ?>
            <a href="<?php echo $base_url . $i; ?>" class="page-link <?php echo $i === $page ? 'active' : ''; ?>">
                <?php echo $i; ?>
            </a>
        <?php endfor; ?>

        <?php if ($end_page < $total_pages): ?>
            <?php if ($end_page < $total_pages - 1): ?>
                <span class="page-dots">...</span>
            <?php endif; ?>
            <a href="<?php echo $base_url . $total_pages; ?>" class="page-link"><?php echo $total_pages; ?></a>
        <?php endif; ?>

        <?php if ($page < $total_pages): ?>
            <a href="<?php echo $base_url . ($page + 1); ?>" class="page-link">Next &raquo;</a>
        <?php endif; ?>
                </div>
                <div class="pagination-info">
                    Showing page <?php echo $page; ?> of <?php echo $total_pages; ?> (<?php echo $total_students; ?> total students)
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Edit Student Modal -->
<div id="editStudentModal" class="modal">
    <div class="modal-backdrop"></div>
    <div class="modal-content">
        <div class="modal-header">
            <div class="modal-title">
                <i class="fas fa-user-edit"></i>
                <h3>Edit Student User</h3>
            </div>
            <button type="button" class="modal-close" onclick="closeEditModal()" aria-label="Close modal">
                <i class="fas fa-times"></i>
            </button>
        </div>

        <div class="modal-body">
            <form method="POST" class="user-form" id="editStudentForm">
                <input type="hidden" name="action" value="update_student">
                <input type="hidden" name="user_id" id="edit_user_id">

                <div class="form-section">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="edit_username" class="form-label">
                                <i class="fas fa-at"></i>
                                Username *
                            </label>
                            <div class="input-wrapper">
                                <input type="text" id="edit_username" name="username" required
                                       class="form-input" placeholder="Enter username">
                                <div class="input-icon">
                                    <i class="fas fa-at"></i>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="edit_password" class="form-label">
                                <i class="fas fa-lock"></i>
                                New Password
                            </label>
                            <div class="input-wrapper">
                                <input type="password" id="edit_password" name="password"
                                       class="form-input" placeholder="Leave blank to keep current">
                                <div class="input-icon">
                                    <i class="fas fa-key"></i>
                                </div>
                            </div>
                            <small class="form-hint">
                                Password must be at least 8 characters with uppercase, lowercase, number, and special character
                            </small>
                        </div>
                    </div>
                </div>

                <div class="modal-footer">
                    <div class="form-actions">
                        <button type="button" onclick="closeEditModal()" class="btn-modal btn-secondary">
                            <i class="fas fa-times"></i>
                            Cancel
                        </button>
                        <button type="submit" class="btn-modal btn-primary">
                            <i class="fas fa-save"></i>
                            Update Student
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
    .student-users-container {
        padding: 20px;
    }

    .page-header {
        margin-bottom: 30px;
    }

    .page-header h2 {
        color: #2c3e50;
        margin-bottom: 10px;
        font-size: 2em;
    }

    .page-header p {
        color: #7f8c8d;
        margin: 0;
        font-size: 1.1em;
    }

    .table-search-container {
        display: flex;
        justify-content: flex-end;
        margin-bottom: 15px;
    }

    .table-search {
        padding: 8px 12px;
        border: 1px solid #ddd;
        border-radius: 5px;
        font-size: 0.9em;
        width: 250px;
    }

    .action-buttons {
        display: flex;
        gap: 5px;
        flex-wrap: wrap;
    }

    .status-badge {
        padding: 4px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: 500;
    }

    .status-badge.active { background-color: #2ecc71; color: white; }
    .status-badge.inactive { background-color: #e74c3c; color: white; }

    /* Enhanced Modal Styles */
    .modal {
        position: fixed;
        z-index: 9999;
        left: 0;
        top: 0;
        width: 100%;
        height: 100%;
        display: flex;
        align-items: center;
        justify-content: center;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s ease;
    }

    .modal.show {
        opacity: 1;
        visibility: visible;
    }

    .modal-backdrop {
        position: absolute;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.4);
        backdrop-filter: blur(1px);
    }

    .modal-content {
        background: white;
        border-radius: 16px;
        width: 90%;
        max-width: 600px;
        max-height: 90vh;
        overflow: hidden;
        box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
        transform: scale(0.9) translateY(-20px);
        transition: all 0.3s ease;
        position: relative;
        z-index: 10000;
    }

    .modal.show .modal-content {
        transform: scale(1) translateY(0);
    }

    .modal-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        padding: 24px 30px;
        display: flex;
        justify-content: space-between;
        align-items: center;
        border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    }

    .modal-title {
        display: flex;
        align-items: center;
        gap: 12px;
        color: white;
    }

    .modal-title i {
        font-size: 1.5em;
        opacity: 0.9;
    }

    .modal-title h3 {
        margin: 0;
        font-size: 1.4em;
        font-weight: 600;
    }

    .modal-close {
        background: rgba(255, 255, 255, 0.2);
        border: none;
        color: white;
        font-size: 1.2em;
        width: 40px;
        height: 40px;
        border-radius: 50%;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: all 0.3s ease;
    }

    .modal-close:hover {
        background: rgba(255, 255, 255, 0.3);
        transform: rotate(90deg);
    }

    .modal-body {
        padding: 30px;
        max-height: calc(90vh - 140px);
        overflow-y: auto;
    }

    .modal-footer {
        padding: 20px 30px;
        background: #f8f9fa;
        border-top: 1px solid #e9ecef;
        display: flex;
        justify-content: flex-end;
    }

    /* Enhanced Form Styles for Modal */
    .form-section {
        margin-bottom: 0;
    }

    .form-label {
        display: flex;
        align-items: center;
        gap: 8px;
        font-weight: 600;
        color: #374151;
        margin-bottom: 8px;
        font-size: 0.95em;
    }

    .form-label i {
        color: #6b7280;
        width: 16px;
    }

    .input-wrapper {
        position: relative;
        display: flex;
        align-items: center;
    }

    .form-input {
        width: 100%;
        padding: 12px 16px;
        border: 2px solid #e5e7eb;
        border-radius: 8px;
        font-size: 1em;
        transition: all 0.3s ease;
        background: white;
    }

    .form-input:focus {
        outline: none;
        border-color: #667eea;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }

    .input-icon {
        position: absolute;
        right: 12px;
        color: #9ca3af;
        pointer-events: none;
        font-size: 0.9em;
    }

    .form-hint {
        display: block;
        margin-top: 6px;
        font-size: 0.85em;
        color: #6b7280;
        font-style: italic;
    }

    /* Modal Button Styles */
    .btn-modal {
        padding: 12px 24px;
        border: none;
        border-radius: 8px;
        font-size: 0.95em;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 8px;
        min-width: 120px;
        justify-content: center;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .btn-modal:focus {
        outline: none;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.3);
    }

    .btn-primary {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
    }

    .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    }

    .btn-secondary {
        background: #6b7280;
        color: white;
        box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
    }

    .btn-secondary:hover {
        background: #4b5563;
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(107, 114, 128, 0.4);
    }

    /* Form Actions in Modal */
    .form-actions {
        display: flex;
        gap: 12px;
        justify-content: flex-end;
        margin: 0;
    }

    /* Pagination Styles */
    .pagination-container {
        margin-top: 20px;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 10px;
    }

    .pagination {
        display: flex;
        justify-content: center;
        align-items: center;
        gap: 5px;
        flex-wrap: wrap;
    }

    .page-link {
        padding: 8px 12px;
        border: 1px solid #ddd;
        background-color: #fff;
        color: #333;
        text-decoration: none;
        border-radius: 4px;
        transition: all 0.3s ease;
        font-size: 0.9em;
        min-width: 35px;
        text-align: center;
    }

    .page-link:hover {
        background-color: #f8f9fa;
        border-color: #adb5bd;
    }

    .page-link.active {
        background-color: #007bff;
        color: white;
        border-color: #007bff;
    }

    .page-dots {
        padding: 8px 4px;
        color: #6c757d;
        font-size: 0.9em;
    }

    .pagination-info {
        font-size: 0.85em;
        color: #6c757d;
        margin-top: 5px;
    }

    @media (max-width: 768px) {
        .table-search {
            width: 100%;
        }

        .modal-content {
            width: 95%;
            margin: 10% auto;
        }

        .modal-backdrop {
            background: rgba(0, 0, 0, 0.3);
            backdrop-filter: none;
        }

        .action-buttons {
            flex-direction: column;
        }

        .pagination {
            gap: 2px;
        }

        .page-link {
            padding: 6px 8px;
            font-size: 0.8em;
            min-width: 30px;
        }
    }
</style>

<script>
    // Edit student function
    function editStudent(student) {
        document.getElementById('edit_user_id').value = student.id;
        document.getElementById('edit_username').value = student.username;

        const modal = document.getElementById('editStudentModal');
        modal.classList.add('show');
        document.body.style.overflow = 'hidden'; // Prevent background scrolling
    }

    // Close edit modal
    function closeEditModal() {
        const modal = document.getElementById('editStudentModal');
        modal.classList.remove('show');
        document.body.style.overflow = ''; // Restore scrolling

        // Clear form after animation completes
        setTimeout(() => {
            document.getElementById('editStudentForm').reset();
        }, 300);
    }

    // Close modal when clicking outside
    window.onclick = function(event) {
        const modal = document.getElementById('editStudentModal');
        if (event.target === modal || event.target.classList.contains('modal-backdrop')) {
            closeEditModal();
        }
    }

    // Close modal on Escape key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            const modal = document.getElementById('editStudentModal');
            if (modal.classList.contains('show')) {
                closeEditModal();
            }
        }
    });

    // Search functionality with debouncing
    let searchTimeout;
    document.querySelector('.table-search').addEventListener('input', function() {
        clearTimeout(searchTimeout);
        const searchTerm = this.value.toLowerCase();
        const rows = document.querySelectorAll('#studentsTable tbody tr');

        searchTimeout = setTimeout(() => {
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                row.style.display = text.includes(searchTerm) ? '' : 'none';
            });
        }, 300); // Debounce search for better performance
    });

    // Password strength validation
    document.getElementById('edit_password').addEventListener('input', function() {
        const password = this.value;
        const strengthIndicator = document.getElementById('password-strength');

        if (password.length === 0) {
            this.style.borderColor = '#e5e7eb';
            return;
        }

        // Strong password regex: at least 8 chars, uppercase, lowercase, number, special char
        const strongRegex = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/;

        if (strongRegex.test(password)) {
            this.style.borderColor = '#10b981'; // Green for strong
        } else {
            this.style.borderColor = '#ef4444'; // Red for weak
        }
    });

    // Form validation feedback
    document.getElementById('editStudentForm').addEventListener('submit', function(e) {
        const requiredFields = this.querySelectorAll('[required]');
        let isValid = true;

        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                field.style.borderColor = '#ef4444';
                isValid = false;
            } else {
                field.style.borderColor = '#10b981';
            }
        });

        // Validate password if provided
        const passwordField = document.getElementById('edit_password');
        if (passwordField.value && !/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/.test(passwordField.value)) {
            passwordField.style.borderColor = '#ef4444';
            isValid = false;
            alert('Password must be at least 8 characters long and contain at least one uppercase letter, one lowercase letter, one number, and one special character.');
        }

        if (!isValid) {
            e.preventDefault();
            // Show error message
            const errorMsg = document.createElement('div');
            errorMsg.className = 'alert alert-error';
            errorMsg.textContent = 'Please fill in all required fields and ensure password meets requirements.';
            errorMsg.style.position = 'fixed';
            errorMsg.style.top = '20px';
            errorMsg.style.right = '20px';
            errorMsg.style.zIndex = '9999';
            document.body.appendChild(errorMsg);

            setTimeout(() => {
                errorMsg.remove();
            }, 3000);
        }
    });

    // Reset field border colors on input
    document.querySelectorAll('.form-input').forEach(field => {
        field.addEventListener('input', function() {
            this.style.borderColor = '#e5e7eb';
        });
    });

    // Deactivate student function
    function deactivateStudent(userId) {
        if (confirm('Are you sure you want to deactivate this student user? They will no longer be able to log in.')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="deactivate_student">
                <input type="hidden" name="user_id" value="${userId}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }

    // Delete student function
    function deleteStudent(userId) {
        if (confirm('Are you sure you want to permanently delete this student user? This action cannot be undone and will also remove their student record.')) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = `
                <input type="hidden" name="action" value="delete_student">
                <input type="hidden" name="user_id" value="${userId}">
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }
</script>

<?php
include '../includes/admin_footer.php';
?>
