<?php
session_start();
require_once '../config.php';
require_once '../includes/enhanced_access_control_fixed.php';

// Check if user has permission to manage roles
requirePermission('role_management');

// Handle form submissions
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();

        if (isset($_POST['action'])) {
            $action = $_POST['action'];

            if ($action === 'add_role' || $action === 'edit_role') {
                $role_name = trim($_POST['role_name']);
                $description = trim($_POST['description']);
                $permissions = isset($_POST['permissions']) ? $_POST['permissions'] : [];

                if (empty($role_name)) {
                    throw new Exception('Role name is required.');
                }

                if ($action === 'add_role') {
                    // Check if role name already exists
                    $stmt = $pdo->prepare("SELECT id FROM roles WHERE role_name = ?");
                    $stmt->execute([$role_name]);
                    if ($stmt->fetch()) {
                        throw new Exception('Role name already exists.');
                    }

                    // Insert new role
                    $stmt = $pdo->prepare("INSERT INTO roles (role_name, description) VALUES (?, ?)");
                    $stmt->execute([$role_name, $description]);
                    $role_id = $pdo->lastInsertId();

                    $message = 'Role added successfully.';
                } else {
                    // Edit existing role
                    $role_id = $_POST['role_id'];
                    $stmt = $pdo->prepare("UPDATE roles SET role_name = ?, description = ? WHERE id = ?");
                    $stmt->execute([$role_name, $description, $role_id]);

                    // Remove existing permissions
                    $stmt = $pdo->prepare("DELETE FROM role_permissions WHERE role_id = ?");
                    $stmt->execute([$role_id]);

                    $message = 'Role updated successfully.';
                }

                // Add permissions
                if (!empty($permissions)) {
                    $stmt = $pdo->prepare("INSERT INTO role_permissions (role_id, permission_id) VALUES (?, ?)");
                    foreach ($permissions as $permission_id) {
                        $stmt->execute([$role_id, $permission_id]);
                    }
                }

            } elseif ($action === 'delete_role') {
                $role_id = $_POST['role_id'];

                // Delete role (cascade will handle role_permissions)
                $stmt = $pdo->prepare("DELETE FROM roles WHERE id = ?");
                $stmt->execute([$role_id]);

                $message = 'Role deleted successfully.';
            }
        }

        $pdo->commit();
        $message_type = 'success';

    } catch (Exception $e) {
        $pdo->rollBack();
        $message = 'Error: ' . $e->getMessage();
        $message_type = 'error';
    }
}

// Fetch all roles
$roles = $pdo->query("SELECT * FROM roles ORDER BY role_name")->fetchAll();

// Fetch all permissions grouped by module
$permissions = $pdo->query("SELECT * FROM permissions ORDER BY module, permission_name")->fetchAll();

// Group permissions by panels
$panels = [
    'Admin Panel' => ['dashboard_access', 'user_management', 'role_management', 'enrollment_management', 'student_management', 'student_logins', 'student_promotion', 'class_management', 'course_management', 'subject_view', 'teacher_management', 'librarian_management', 'non_teaching_management', 'director_management', 'head_teacher_management', 'staff_leave_management', 'academic_years', 'terms_management', 'levels_management', 'reports_analytics', 'transportation_management', 'grading_system', 'settings'],
    'Accounting System' => ['fee_management', 'assign_fees', 'payment_management', 'fee_reports', 'payroll_management', 'budget_vs_actual', 'cash_flow_statement', 'profit_loss_report', 'balance_sheet_report', 'custom_report_builder', 'export_financial_reports', 'accountant_management'],
    'Library System' => ['inventory_management', 'facilities_management'],
    'Teacher Panel' => ['dashboard_access', 'enrollment_management', 'student_management', 'class_management', 'subject_view', 'reports_analytics', 'grading_system']
];

$grouped_permissions = [];
foreach ($panels as $panel => $panel_permissions) {
    $grouped_permissions[$panel] = array_filter($permissions, function($perm) use ($panel_permissions) {
        return in_array($perm['permission_name'], $panel_permissions);
    });
}

include '../includes/admin_header.php';
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Manage Roles & Permissions</h3>
                    <div class="card-tools">
                        <button type="button" class="btn btn-primary" data-toggle="modal" data-target="#roleModal" onclick="openAddModal()">
                            <i class="fas fa-plus"></i> Add New Role
                        </button>
                    </div>
                </div>

                <?php if ($message): ?>
                    <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible">
                        <button type="button" class="close" data-dismiss="alert">&times;</button>
                        <?php echo htmlspecialchars($message); ?>
                    </div>
                <?php endif; ?>

                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Role Name</th>
                                    <th>Description</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($roles as $role): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($role['id']); ?></td>
                                        <td><?php echo htmlspecialchars($role['role_name']); ?></td>
                                        <td><?php echo htmlspecialchars($role['description']); ?></td>
                                        <td>
                                            <button type="button" class="btn btn-sm btn-warning" onclick="openEditModal(<?php echo $role['id']; ?>)">
                                                <i class="fas fa-edit"></i> Edit
                                            </button>
                                            <button type="button" class="btn btn-sm btn-danger" onclick="deleteRole(<?php echo $role['id']; ?>, '<?php echo htmlspecialchars($role['role_name']); ?>')">
                                                <i class="fas fa-trash"></i> Delete
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Role Modal -->
<div class="modal fade" id="roleModal" tabindex="-1" role="dialog" aria-labelledby="roleModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <form id="roleForm" method="POST">
                <div class="modal-header">
                    <h5 class="modal-title" id="roleModalLabel">Add/Edit Role</h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="action" id="action" value="add_role">
                    <input type="hidden" name="role_id" id="role_id" value="">

                    <div class="form-group">
                        <label for="role_name">Role Name *</label>
                        <input type="text" class="form-control" id="role_name" name="role_name" required>
                    </div>

                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                    </div>

                    <div class="form-group">
                        <label>Permissions</label>
                        <div class="row">
                            <?php foreach ($grouped_permissions as $panel => $panel_perms): ?>
                                <div class="col-md-6">
                                    <h6><?php echo htmlspecialchars($panel); ?></h6>
                                    <?php foreach ($panel_perms as $perm): ?>
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" name="permissions[]" value="<?php echo $perm['id']; ?>" id="perm_<?php echo $perm['id']; ?>">
                                            <label class="form-check-label" for="perm_<?php echo $perm['id']; ?>">
                                                <?php echo htmlspecialchars($perm['permission_name']); ?>
                                                <?php if ($perm['description']): ?>
                                                    <small class="text-muted">(<?php echo htmlspecialchars($perm['description']); ?>)</small>
                                                <?php endif; ?>
                                            </label>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Role</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" role="dialog" aria-labelledby="deleteModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteModalLabel">Confirm Delete</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                Are you sure you want to delete the role "<span id="deleteRoleName"></span>"? This action cannot be undone.
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <form id="deleteForm" method="POST" style="display: inline;">
                    <input type="hidden" name="action" value="delete_role">
                    <input type="hidden" name="role_id" id="deleteRoleId" value="">
                    <button type="submit" class="btn btn-danger">Delete</button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
function openAddModal() {
    document.getElementById('action').value = 'add_role';
    document.getElementById('role_id').value = '';
    document.getElementById('role_name').value = '';
    document.getElementById('description').value = '';
    document.querySelectorAll('input[name="permissions[]"]').forEach(cb => cb.checked = false);
    document.getElementById('roleModalLabel').textContent = 'Add New Role';
    $('#roleModal').modal('show');
}

function openEditModal(roleId) {
    document.getElementById('action').value = 'edit_role';
    document.getElementById('role_id').value = roleId;
    document.getElementById('roleModalLabel').textContent = 'Edit Role';

    // Fetch role data
    fetch(`get_role_permissions.php?role_id=${roleId}`)
        .then(response => response.json())
        .then(data => {
            document.getElementById('role_name').value = data.role.role_name;
            document.getElementById('description').value = data.role.description;
            document.querySelectorAll('input[name="permissions[]"]').forEach(cb => {
                cb.checked = data.permissions.includes(parseInt(cb.value));
            });
            $('#roleModal').modal('show');
        });
}

function deleteRole(roleId, roleName) {
    document.getElementById('deleteRoleId').value = roleId;
    document.getElementById('deleteRoleName').textContent = roleName;
    $('#deleteModal').modal('show');
}
</script>

<?php include '../includes/admin_footer.php'; ?>
