<?php
ob_start();
require_once '../config.php';

$message = '';
$success = false;

// Create teacher_classes table if not exists
$pdo->exec("CREATE TABLE IF NOT EXISTS teacher_classes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    teacher_id INT NOT NULL,
    class_id INT NOT NULL,
    UNIQUE KEY unique_teacher_class (teacher_id, class_id),
    FOREIGN KEY (teacher_id) REFERENCES teachers(id) ON DELETE CASCADE,
    FOREIGN KEY (class_id) REFERENCES classes(id) ON DELETE CASCADE
)");

// Create teacher_courses table if not exists
$pdo->exec("CREATE TABLE IF NOT EXISTS teacher_courses (
    id INT AUTO_INCREMENT PRIMARY KEY,
    teacher_id INT NOT NULL,
    course_id INT NOT NULL,
    UNIQUE KEY unique_teacher_course (teacher_id, course_id),
    FOREIGN KEY (teacher_id) REFERENCES teachers(id) ON DELETE CASCADE,
    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE
)");

// Fetch all teaching staff
$stmt = $pdo->query("SELECT id, staff_id, first_name, middle_name, last_name, job_position FROM staff WHERE role = 'teaching' ORDER BY last_name, first_name");
$teachers = $stmt->fetchAll();

// Fetch all courses
$stmt = $pdo->query("SELECT id, name FROM courses ORDER BY name");
$courses = $stmt->fetchAll();

// Fetch all classes
$stmt = $pdo->query("SELECT id, name FROM classes ORDER BY name");
$classes = $stmt->fetchAll();

// Fetch existing course assignments
$teacher_courses = [];
$stmt = $pdo->query("SELECT teacher_id, course_id FROM teacher_courses");
while ($row = $stmt->fetch()) {
    $teacher_courses[$row['teacher_id']][] = $row['course_id'];
}

// Fetch existing class assignments
$teacher_classes = [];
$stmt = $pdo->query("SELECT teacher_id, class_id FROM teacher_classes");
while ($row = $stmt->fetch()) {
    $teacher_classes[$row['teacher_id']][] = $row['class_id'];
}

// Handle course assignment form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['submit_courses'])) {
    try {
        $teacher_id = $_POST['teacher_id'];
        $selected_courses = $_POST['courses'] ?? [];

        if (empty($selected_courses)) {
            $message = 'No courses selected for assignment.';
            $success = false;
        } else {
            // Remove all current assignments for this teacher from teacher_courses table
            $stmt = $pdo->prepare("DELETE FROM teacher_courses WHERE teacher_id = ?");
            $stmt->execute([$teacher_id]);

            // Insert the selected courses into teacher_courses table
            $stmt = $pdo->prepare("INSERT INTO teacher_courses (teacher_id, course_id) VALUES (?, ?)");
            foreach ($selected_courses as $course_id) {
                $stmt->execute([$teacher_id, $course_id]);
            }

            // Redirect to avoid POST resubmission on refresh
            header("Location: assign_class.php?success=courses&tab=" . ($_POST['tab'] ?? 'courses') . "&teacher_id=" . $teacher_id);
            exit;
        }
    } catch (Exception $e) {
        $message = 'Error assigning courses: ' . $e->getMessage();
        $success = false;
    }
}

// Handle class assignment form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['submit_classes'])) {
    try {
        $teacher_id = $_POST['teacher_id'];
        $selected_classes = $_POST['classes'] ?? [];

        // First, remove all current class assignments for this teacher
        $stmt = $pdo->prepare("DELETE FROM teacher_classes WHERE teacher_id = ?");
        $stmt->execute([$teacher_id]);

        // Then assign the selected classes
        if (!empty($selected_classes)) {
            $stmt = $pdo->prepare("INSERT INTO teacher_classes (teacher_id, class_id) VALUES (?, ?)");
            foreach ($selected_classes as $class_id) {
                $stmt->execute([$teacher_id, $class_id]);
            }
        }

        // Redirect to avoid POST resubmission on refresh
        header("Location: assign_class.php?success=classes&tab=" . ($_POST['tab'] ?? 'classes') . "&teacher_id=" . $teacher_id);
        exit;
    } catch (Exception $e) {
        $message = 'Error assigning classes: ' . $e->getMessage();
        $success = false;
    }
}

// Check for success message from GET parameter
if (isset($_GET['success'])) {
    if ($_GET['success'] == 'courses') {
        $message = 'Courses assigned to teacher successfully.';
        $success = true;
    } elseif ($_GET['success'] == 'classes') {
        $message = 'Classes assigned to teacher successfully.';
        $success = true;
    }
}

// Get selected teacher for courses display
$selected_teacher_id = $_GET['teacher_id'] ?? $_POST['teacher_id'] ?? null;
$selected_courses_for_teacher = $selected_teacher_id && isset($teacher_courses[$selected_teacher_id]) ? $teacher_courses[$selected_teacher_id] : [];

// Get selected teacher for classes display
$selected_teacher_class_id = $_GET['teacher_id'] ?? $_POST['teacher_id'] ?? null;
$selected_classes_for_teacher = $selected_teacher_class_id && isset($teacher_classes[$selected_teacher_class_id]) ? $teacher_classes[$selected_teacher_class_id] : [];

// When a teacher is selected in classes tab, also select the same for courses
if ($selected_teacher_class_id) {
    $selected_teacher_id = $selected_teacher_class_id;
    $selected_courses_for_teacher = $selected_teacher_class_id && isset($teacher_courses[$selected_teacher_class_id]) ? $teacher_courses[$selected_teacher_class_id] : [];
}

// Determine active tab
$active_tab = $_GET['tab'] ?? $_POST['tab'] ?? 'courses';

include '../includes/admin_header.php';
?>

<h2>Assign Courses and Classes to Teachers</h2>

<?php if ($message && $success): ?>
    <div class="alert alert-success" role="alert">
        <?php echo htmlspecialchars($message); ?>
    </div>
<?php endif; ?>

<form method="post" action="assign_class.php">
    <div class="form-group">
        <label for="teacher_id">Select Teacher:</label>
        <select id="teacher_id" name="teacher_id" onchange="this.form.submit()" required>
            <option value="">-- Select Teacher --</option>
            <?php foreach ($teachers as $teacher): ?>
                <option value="<?php echo $teacher['id']; ?>" <?php echo ($selected_teacher_id == $teacher['id']) ? 'selected' : ''; ?>>
                    <?php echo htmlspecialchars(trim($teacher['first_name'] . ' ' . $teacher['middle_name'] . ' ' . $teacher['last_name'])); ?> (<?php echo htmlspecialchars($teacher['job_position']); ?>)
                </option>
            <?php endforeach; ?>
        </select>
    </div>

    <?php if ($selected_teacher_id): ?>
        <div class="assignment-container">
            <div class="assignment-section">
                <h3>Assign Courses</h3>
                <div class="courses-list">
                    <?php foreach ($courses as $course): ?>
                        <label>
                            <input type="checkbox" name="courses[]" value="<?php echo $course['id']; ?>" <?php echo in_array($course['id'], $selected_courses_for_teacher) ? 'checked' : ''; ?>>
                            <?php echo htmlspecialchars($course['name']); ?>
                        </label>
                    <?php endforeach; ?>
                </div>
                <button type="submit" name="submit_courses" value="1" class="btn btn-primary">Save Course Assignments</button>
            </div>
            <div class="assignment-section">
                <h3>Assign Classes</h3>
                <div class="classes-list">
                    <?php foreach ($classes as $class): ?>
                        <label>
                            <input type="checkbox" name="classes[]" value="<?php echo $class['id']; ?>" <?php echo in_array($class['id'], $selected_classes_for_teacher) ? 'checked' : ''; ?>>
                            <?php echo htmlspecialchars($class['name']); ?>
                        </label>
                    <?php endforeach; ?>
                </div>
                <button type="submit" name="submit_classes" value="1" class="btn btn-primary">Save Class Assignments</button>
            </div>
        </div>
    <?php endif; ?>
</form>

<h3>Current Assignments</h3>

<table class="fees-table">
    <thead>
        <tr>
            <th>Teacher</th>
            <th>Position</th>
            <th>Classes</th>
            <th>Assigned Courses</th>
        </tr>
    </thead>
    <tbody>
        <?php foreach ($teachers as $teacher): ?>
            <tr>
                <td><?php echo htmlspecialchars(trim($teacher['first_name'] . ' ' . $teacher['middle_name'] . ' ' . $teacher['last_name'])); ?></td>
                <td><?php echo htmlspecialchars($teacher['job_position']); ?></td>
                <td>
                    <?php
                    $stmt = $pdo->prepare("SELECT cl.name FROM classes cl JOIN teacher_classes tc ON cl.id = tc.class_id WHERE tc.teacher_id = ? ORDER BY cl.name");
                    $stmt->execute([$teacher['id']]);
                    $assigned_classes = $stmt->fetchAll(PDO::FETCH_COLUMN);
                    echo htmlspecialchars(implode(', ', $assigned_classes));
                    ?>
                </td>
                <td>
                    <?php
                    $stmt = $pdo->prepare("SELECT COUNT(*) FROM teacher_courses WHERE teacher_id = ?");
                    $stmt->execute([$teacher['id']]);
                    $count = $stmt->fetchColumn();
                    if ($count > 0) {
                        echo '<button type="button" class="btn btn-small btn-info" onclick="viewCourses(' . $teacher['id'] . ', \'' . htmlspecialchars(addslashes(trim($teacher['first_name'] . ' ' . $teacher['middle_name'] . ' ' . $teacher['last_name']))) . '\')">View (' . $count . ')</button>';
                    } else {
                        echo 'None';
                    }
                    ?>
                </td>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

<!-- Modal for viewing assigned courses -->
<div class="modal fade" id="viewCoursesModal" tabindex="-1" role="dialog" aria-labelledby="viewCoursesModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewCoursesModalLabel">Assigned Courses</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <p id="teacherName"></p>
                <ul id="coursesList" class="list-group"></ul>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<style>
.assignment-container {
    display: flex;
    gap: 20px;
    margin-top: 20px;
}

.assignment-section {
    flex: 1;
    background: white;
    border-radius: 10px;
    padding: 20px;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}

.assignment-section h3 {
    margin-bottom: 15px;
    color: #495057;
    font-weight: 600;
}

.courses-list, .classes-list {
    max-height: 400px;
    overflow-y: auto;
    border: 1px solid #ddd;
    border-radius: 5px;
    padding: 10px;
    background: #fff;
}

.courses-list label, .classes-list label {
    display: block;
    padding: 8px;
    border-bottom: 1px solid #eee;
    cursor: pointer;
}

.courses-list label:hover, .classes-list label:hover {
    background: #f8f9fa;
}

.courses-list input[type="checkbox"], .classes-list input[type="checkbox"] {
    margin-right: 10px;
}

.btn-primary {
    background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
    border: none;
    border-radius: 20px;
    padding: 10px 20px;
    font-weight: 500;
    transition: all 0.3s ease;
    margin-top: 15px;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.fees-table {
    background: white;
    border-radius: 10px;
    overflow: hidden;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    margin-top: 30px;
}

.fees-table table {
    width: 100%;
    border-collapse: collapse;
}

.fees-table th,
.fees-table td {
    padding: 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.fees-table th {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    font-weight: 600;
    font-size: 16px;
}

.fees-table tr:hover {
    background: #f8f9fa;
}

.btn-info {
    background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);
    border: none;
    border-radius: 20px;
    padding: 8px 16px;
    font-weight: 500;
    transition: all 0.3s ease;
}

.btn-info:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.modal-content {
    border-radius: 10px;
    border: none;
    box-shadow: 0 10px 30px rgba(0,0,0,0.3);
}

.modal-header {
    background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);
    color: white;
    border-radius: 10px 10px 0 0;
    border-bottom: none;
}

.modal-body {
    padding: 20px;
}

.modal-footer .btn-secondary {
    background: linear-gradient(135deg, #17a2b8 0%, #138496 100%);
    border: none;
    border-radius: 20px;
    padding: 8px 16px;
    font-weight: 500;
    transition: all 0.3s ease;
    color: white;
}

.modal-footer .btn-secondary:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
    background: linear-gradient(135deg, #138496 0%, #17a2b8 100%);
}

.list-group-item {
    border: none;
    border-radius: 5px;
    margin-bottom: 5px;
    background: #f8f9fa;
}
</style>



<script>
function viewCourses(teacherId, teacherName) {
    // Fetch assigned courses via AJAX
    fetch('fetch_teacher_courses.php?teacher_id=' + teacherId)
        .then(response => response.json())
        .then(data => {
            const coursesList = document.getElementById('coursesList');
            const teacherNameElem = document.getElementById('teacherName');
            coursesList.innerHTML = '';
            teacherNameElem.textContent = 'Courses assigned to ' + teacherName + ':';
            if (data.length === 0) {
                const li = document.createElement('li');
                li.className = 'list-group-item';
                li.textContent = 'No courses assigned.';
                coursesList.appendChild(li);
            } else {
                data.forEach(course => {
                    const li = document.createElement('li');
                    li.className = 'list-group-item';
                    li.textContent = course.name;
                    coursesList.appendChild(li);
                });
            }
            $('#viewCoursesModal').modal('show');
        })
        .catch(error => {
            alert('Failed to fetch courses: ' + error);
        });
}
</script>

<?php
include '../includes/admin_footer.php';
?>
