<?php
require_once '../config.php';
include 'includes/accounting_header.php';

// Get filter parameters
$class_filter = $_GET['class_filter'] ?? '';
$search_filter = $_GET['search'] ?? '';

// Build query to get student tuition fee receivable data from journal entries
$query = "
SELECT
    s.id,
    CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as name,
    s.class AS class_name,
    COALESCE(je_data.total_assigned, 0) AS total_fees_assigned,
    COALESCE(je_data.total_paid, 0) AS total_paid,
    GREATEST(0, COALESCE(je_data.total_assigned, 0) - COALESCE(je_data.total_paid, 0)) AS outstanding_balance
FROM students s
LEFT JOIN (
    SELECT
        CASE
            WHEN jel.debit_amount > 0 THEN je.source_id
            WHEN jel.credit_amount > 0 THEN sf.student_id
        END as student_id,
        SUM(CASE WHEN jel.debit_amount > 0 THEN jel.debit_amount ELSE 0 END) as total_assigned,
        SUM(CASE WHEN jel.credit_amount > 0 THEN jel.credit_amount ELSE 0 END) as total_paid
    FROM journal_entries je
    JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
    JOIN chart_of_accounts coa ON jel.account_id = coa.id
    LEFT JOIN student_fees sf ON (
        (jel.debit_amount > 0 AND je.source_id = sf.student_id) OR
        (jel.credit_amount > 0 AND je.source_id = sf.id)
    )
    JOIN fees f ON sf.fee_id = f.id
    WHERE je.status = 'posted'
    AND coa.account_code = '1121'
    AND f.type = 'tuition'
    GROUP BY CASE
        WHEN jel.debit_amount > 0 THEN je.source_id
        WHEN jel.credit_amount > 0 THEN sf.student_id
    END
) je_data ON s.id = je_data.student_id
WHERE COALESCE(je_data.total_assigned, 0) - COALESCE(je_data.total_paid, 0) > 0
";

$params = [];

if (!empty($class_filter)) {
    $query .= " AND s.class = ?";
    $params[] = $class_filter;
}

if (!empty($search_filter)) {
    $query .= " AND CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) LIKE ?";
    $params[] = '%' . $search_filter . '%';
}

$query .= " ORDER BY CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) ASC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$receivables = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate totals
$total_outstanding = array_sum(array_column($receivables, 'outstanding_balance'));
$total_students = count($receivables);

// Get classes for filter
$classes = $pdo->query("SELECT DISTINCT class FROM students ORDER BY class ASC")->fetchAll(PDO::FETCH_COLUMN);

// Get school settings for print
$school_settings = $pdo->query("SELECT school_name, address, phone, email FROM school_settings LIMIT 1")->fetch(PDO::FETCH_ASSOC);
?>

<style>
/* Basic styling for table and modal */
.table-container {
    margin: 20px;
    overflow-x: auto;
}

table {
    width: 100%;
    border-collapse: collapse;
    margin-bottom: 20px;
}

th, td {
    padding: 12px 15px;
    border: 1px solid #ddd;
    text-align: left;
}

#receivables-table th {
    background: white !important;
    color: black !important;
}

.outstanding-amount {
    color: #e74c3c;
    font-weight: bold;
}

.total-row {
    background: #f8f9fa;
    font-weight: bold;
    font-size: 1.1em;
}

/* Filter form styling */
.filter-form {
    background-color: #f9f9f9;
    padding: 20px;
    border-radius: 8px;
    border: 1px solid #ddd;
    margin-bottom: 20px;
    display: flex;
    gap: 15px;
    align-items: center;
    flex-wrap: wrap;
}

.filter-form label {
    font-weight: bold;
    color: #333;
    margin-right: 5px;
}

.filter-form select,
.filter-form input[type="text"] {
    padding: 8px 12px;
    border: 1px solid #ccc;
    border-radius: 4px;
    font-size: 14px;
    min-width: 120px;
}

.filter-form select:focus,
.filter-form input[type="text"]:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 5px rgba(76, 175, 80, 0.3);
}

.filter-form button {
    background-color: #4CAF50;
    color: white;
    border: none;
    padding: 10px 15px;
    cursor: pointer;
    border-radius: 4px;
    font-size: 14px;
    font-weight: bold;
}

.filter-form button:hover {
    background-color: #45a049;
}

.filter-form a {
    text-decoration: none;
    background-color: #f44336;
    color: white;
    padding: 10px 15px;
    border-radius: 4px;
    font-size: 14px;
    font-weight: bold;
}

.filter-form a:hover {
    background-color: #d32f2f;
}

/* Summary cards */
.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.summary-card {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.summary-card .card-icon {
    font-size: 2.5em;
    margin-bottom: 15px;
}

.summary-card h3 {
    color: #2c3e50;
    font-size: 1.8em;
    margin-bottom: 5px;
}

.summary-card p {
    color: #7f8c8d;
    margin: 0;
}

@media (max-width: 768px) {
    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }

    .summary-cards {
        grid-template-columns: 1fr;
    }
}
</style>

<div class="table-container">
    <h2>📊 Student Tuition Fee Receivable Report</h2>
    <p>Outstanding student tuition fees receivable</p>

    <!-- Summary Cards -->
    <div class="summary-cards">
        <div class="summary-card">
            <div class="card-icon">👥</div>
            <h3><?php echo $total_students; ?></h3>
            <p>Students with Outstanding Tuition Balances</p>
        </div>

        <div class="summary-card">
            <div class="card-icon">💰</div>
            <h3>GHC <?php echo number_format($total_outstanding, 2); ?></h3>
            <p>Total Outstanding Tuition Amount</p>
        </div>

        <div class="summary-card">
            <div class="card-icon">📈</div>
            <h3>GHC <?php echo $total_students > 0 ? number_format($total_outstanding / $total_students, 2) : '0.00'; ?></h3>
            <p>Average Outstanding Tuition per Student</p>
        </div>
    </div>

    <!-- Filter Form -->
    <form method="GET" class="filter-form">
        <label>Class:
            <select name="class_filter">
                <option value="">All Classes</option>
                <?php foreach ($classes as $class): ?>
                    <option value="<?php echo htmlspecialchars($class); ?>" <?php echo ($class_filter == $class) ? 'selected' : ''; ?>><?php echo htmlspecialchars($class); ?></option>
                <?php endforeach; ?>
            </select>
        </label>

        <label>Search Student:
            <input type="text" name="search" value="<?php echo htmlspecialchars($search_filter); ?>" placeholder="Student name">
        </label>

        <button type="submit">Filter</button>
        <a href="student_tuition_fee_receivable.php">Clear</a>
        <button type="button" onclick="printReport()" style="background-color: #2196F3;">Print Report</button>
    </form>

    <table id="receivables-table">
        <thead>
            <tr>
                <th>Student Name</th>
                <th>Class</th>
                <th>Total Tuition Fees Assigned (GHC)</th>
                <th>Tuition Amount Paid (GHC)</th>
                <th>Outstanding Tuition Balance (GHC)</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($receivables as $receivable): ?>
            <tr>
                <td><?php echo htmlspecialchars($receivable['name']); ?></td>
                <td><?php echo htmlspecialchars($receivable['class_name']); ?></td>
                <td><?php echo number_format($receivable['total_fees_assigned'], 2); ?></td>
                <td><?php echo number_format($receivable['total_paid'], 2); ?></td>
                <td class="outstanding-amount"><?php echo number_format($receivable['outstanding_balance'], 2); ?></td>
            </tr>
            <?php endforeach; ?>

            <?php if (empty($receivables)): ?>
            <tr>
                <td colspan="5" style="text-align: center; color: #7f8c8d; font-style: italic;">No outstanding tuition receivables found</td>
            </tr>
            <?php else: ?>
            <tr class="total-row">
                <td colspan="4"><strong>Total Outstanding Tuition</strong></td>
                <td class="outstanding-amount"><strong>GHC <?php echo number_format($total_outstanding, 2); ?></strong></td>
            </tr>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<script>
function printReport() {
    // Create a new window for printing
    var printWindow = window.open('', '_blank');

    // Get school settings
    var schoolName = '<?php echo htmlspecialchars($school_settings['school_name'] ?? 'School Name'); ?>';
    var schoolAddress = '<?php echo htmlspecialchars($school_settings['address'] ?? 'School Address'); ?>';
    var schoolPhone = '<?php echo htmlspecialchars($school_settings['phone'] ?? 'School Phone'); ?>';
    var schoolEmail = '<?php echo htmlspecialchars($school_settings['email'] ?? 'School Email'); ?>';

    // Build the print content
    var printContent = `
        <!DOCTYPE html>
        <html>
        <head>
            <title>Student Tuition Fee Receivable Report</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    margin: 20px;
                    line-height: 1.4;
                }
                .header {
                    text-align: center;
                    border-bottom: 2px solid #333;
                    padding-bottom: 20px;
                    margin-bottom: 30px;
                }
                .school-info {
                    margin-bottom: 20px;
                }
                .report-title {
                    font-size: 24px;
                    font-weight: bold;
                    margin: 20px 0;
                    color: #2c3e50;
                }
                .summary-section {
                    display: flex;
                    justify-content: space-around;
                    margin-bottom: 30px;
                    padding: 20px;
                    background: #f8f9fa;
                    border-radius: 8px;
                }
                .summary-item {
                    text-align: center;
                    flex: 1;
                }
                .summary-item h3 {
                    font-size: 28px;
                    margin: 10px 0;
                    color: #2c3e50;
                }
                .summary-item p {
                    color: #7f8c8d;
                    margin: 5px 0;
                    font-size: 14px;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 20px;
                    font-size: 12px;
                }
                th, td {
                    border: 1px solid #ddd;
                    padding: 8px 12px;
                    text-align: left;
                }
                th {
                    background-color: #f8f9fa;
                    font-weight: bold;
                    color: #2c3e50;
                }
                .total-row {
                    background-color: #e9ecef;
                    font-weight: bold;
                }
                .outstanding-amount {
                    color: #dc3545;
                    font-weight: bold;
                }
                .footer {
                    margin-top: 40px;
                    text-align: center;
                    font-size: 12px;
                    color: #6c757d;
                    border-top: 1px solid #dee2e6;
                    padding-top: 20px;
                }
                @media print {
                    body { margin: 0; }
                    .no-print { display: none; }
                }
            </style>
        </head>
        <body>
            <div class="header">
                <div class="school-info">
                    <h1>${schoolName}</h1>
                    <p>${schoolAddress}</p>
                    <p>Phone: ${schoolPhone} | Email: ${schoolEmail}</p>
                </div>
                <h2 class="report-title">Student Tuition Fee Receivable Report</h2>
                <p>Generated on: ${new Date().toLocaleDateString()} ${new Date().toLocaleTimeString()}</p>
            </div>

            <div class="summary-section">
                <div class="summary-item">
                    <h3><?php echo $total_students; ?></h3>
                    <p>Students with Outstanding<br>Tuition Balances</p>
                </div>
                <div class="summary-item">
                    <h3>GHC <?php echo number_format($total_outstanding, 2); ?></h3>
                    <p>Total Outstanding<br>Tuition Amount</p>
                </div>
                <div class="summary-item">
                    <h3>GHC <?php echo $total_students > 0 ? number_format($total_outstanding / $total_students, 2) : '0.00'; ?></h3>
                    <p>Average Outstanding<br>Tuition per Student</p>
                </div>
            </div>

            <table>
                <thead>
                    <tr>
                        <th>Student Name</th>
                        <th>Class</th>
                        <th>Total Tuition Fees Assigned (GHC)</th>
                        <th>Tuition Amount Paid (GHC)</th>
                        <th>Outstanding Tuition Balance (GHC)</th>
                    </tr>
                </thead>
                <tbody>
    `;

    <?php foreach ($receivables as $receivable): ?>
    printContent += `
                    <tr>
                        <td><?php echo htmlspecialchars($receivable['name']); ?></td>
                        <td><?php echo htmlspecialchars($receivable['class_name']); ?></td>
                        <td><?php echo number_format($receivable['total_fees_assigned'], 2); ?></td>
                        <td><?php echo number_format($receivable['total_paid'], 2); ?></td>
                        <td class="outstanding-amount"><?php echo number_format($receivable['outstanding_balance'], 2); ?></td>
                    </tr>
    `;
    <?php endforeach; ?>

    <?php if (!empty($receivables)): ?>
    printContent += `
                    <tr class="total-row">
                        <td colspan="4"><strong>Total Outstanding Tuition</strong></td>
                        <td class="outstanding-amount"><strong>GHC <?php echo number_format($total_outstanding, 2); ?></strong></td>
                    </tr>
    `;
    <?php endif; ?>

    printContent += `
                </tbody>
            </table>

            <div class="footer">
                <p>This report was generated by the School Management System</p>
                <p>Report Date: ${new Date().toLocaleDateString()}</p>
            </div>
        </body>
        </html>
    `;

    // Write content to the new window
    printWindow.document.write(printContent);
    printWindow.document.close();

    // Wait for content to load then print
    printWindow.onload = function() {
        printWindow.print();
        printWindow.close();
    };
}
</script>

<style>
@media print {
    .filter-form, .summary-cards {
        display: none;
    }

    .table-container {
        margin: 0;
    }

    h2 {
        text-align: center;
        margin-bottom: 20px;
    }

    table {
        font-size: 12px;
    }

    th, td {
        padding: 8px;
    }
}
</style>

<?php
include 'includes/accounting_footer.php';
?>
