<?php
session_start();

$receipt = null;

// Check if receipt data is in session (from salary payment)
if (isset($_SESSION['salary_receipt_data'])) {
    $receipt = $_SESSION['salary_receipt_data'];
    // Clear receipt data after use
    unset($_SESSION['salary_receipt_data']);
}
// Check if payslip ID is provided (from payslips view)
elseif (isset($_GET['payslip'])) {
    $payslip_id = $_GET['payslip'];

    // Fetch payslip details from database
    require_once '../config.php';

    try {
        $stmt = $pdo->prepare("
            SELECT p.*, pr.payroll_run_date, pr.id as payroll_run_id,
                   CONCAT(s.first_name, ' ', s.last_name) as employee_name,
                   p.net_salary as amount, p.payment_status, p.paid_at, p.payment_method,
                   p.payment_comments, u.username as paid_by_name,
                   CONCAT(pay.first_name, ' ', pay.last_name) as processed_by_name
            FROM payslips p
            JOIN payroll_runs pr ON p.payroll_run_id = pr.id
            LEFT JOIN staff s ON p.staff_id = s.id
            LEFT JOIN users u ON p.paid_by = u.id
            LEFT JOIN users pay ON p.paid_by = pay.id
            WHERE p.id = ? AND p.payment_status = 'paid'
        ");
        $stmt->execute([$payslip_id]);
        $payslip_data = $stmt->fetch();

        if ($payslip_data) {
            // Format receipt data
            $receipt = [
                'receipt_number' => 'SAL-' . str_pad($payslip_data['id'], 6, '0', STR_PAD_LEFT),
                'employee_name' => $payslip_data['employee_name'],
                'payroll_run' => 'Run #' . $payslip_data['payroll_run_id'],
                'payroll_date' => $payslip_data['payroll_run_date'],
                'amount' => $payslip_data['amount'],
                'payment_date' => $payslip_data['paid_at'],
                'payment_method' => $payslip_data['payment_method'],
                'comments' => $payslip_data['payment_comments'],
                'processed_by' => $payslip_data['processed_by_name'] ?: $payslip_data['paid_by_name'],
                'status' => 'Paid'
            ];
        }
    } catch (Exception $e) {
        // Handle database error
    }
}

if (!$receipt) {
    echo "No receipt data available.";
    exit;
}

// Fetch school settings
require_once '../config.php';
$school_name = "School Name";
$school_address = "School Address, City, Ghana";

try {
    $stmt = $pdo->prepare("SELECT school_name, address FROM school_settings LIMIT 1");
    $stmt->execute();
    $school_info = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($school_info) {
        $school_name = htmlspecialchars($school_info['school_name']);
        $school_address = htmlspecialchars($school_info['address']);
    }
} catch (Exception $e) {
    // Use default values if database query fails
    $school_name = "School Name";
    $school_address = "School Address, City, Ghana";
}

$report_date = date('Y-m-d');
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Salary Payment Receipt - <?php echo htmlspecialchars($receipt['receipt_number'] ?? ''); ?></title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 15px 20px;
            line-height: 1.3;
            color: #222;
            background-color: #fff;
            max-width: 600px;
            margin-left: auto;
            margin-right: auto;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
            font-size: 14px;
        }

        .receipt-container {
            max-width: 100%;
        }

        /* Header Section */
        .header-section {
            display: flex;
            justify-content: center;
            align-items: center;
            flex-direction: column;
            margin-bottom: 30px;
            border-bottom: 3px solid #2E7D32;
            padding-bottom: 15px;
            text-align: center;
        }

        .school-info h1 {
            margin: 0 0 8px 0;
            font-size: 22px;
            color: #2E7D32;
            font-weight: 700;
            letter-spacing: 1px;
        }

        .school-info p {
            margin: 0;
            font-size: 14px;
            color: #555;
        }

        .receipt-title h2 {
            margin: 10px 0 5px 0;
            font-size: 20px;
            color: #2E7D32;
            font-weight: 700;
            letter-spacing: 1px;
        }

        .receipt-number {
            margin: 3px 0;
            font-size: 14px;
            font-weight: 600;
            color: #333;
        }

        .date {
            margin: 3px 0;
            font-size: 12px;
            color: #666;
        }

        /* Information Sections */
        .info-section, .payment-section, .summary-section {
            margin-bottom: 20px;
            page-break-inside: avoid;
        }

        .info-section h3, .payment-section h3, .summary-section h3 {
            margin: 0 0 10px 0;
            font-size: 16px;
            color: #2E7D32;
            font-weight: 700;
            border-bottom: 2px solid #e9ecef;
            padding-bottom: 3px;
        }

        .info-grid, .summary-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 8px;
        }

        .info-row, .summary-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 0;
            border-bottom: 1px solid #f0f0f0;
        }

        .info-row:last-child, .summary-row:last-child {
            border-bottom: none;
        }

        .info-row label, .summary-row label {
            font-weight: 600;
            color: #555;
            min-width: 140px;
        }

        .info-row span, .summary-row span {
            font-weight: 500;
            color: #333;
            text-align: right;
        }

        .status-row span.status {
            font-weight: 700;
            padding: 4px 12px;
            border-radius: 4px;
            background: #28a745;
            color: white;
        }

        /* Payment Details */
        .amount-display {
            background: #f8f9fa;
            border: 2px solid #2E7D32;
            border-radius: 6px;
            padding: 15px;
            text-align: center;
        }

        .amount-row {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 15px;
        }

        .amount-label {
            font-size: 16px;
            font-weight: 600;
            color: #2E7D32;
        }

        .amount-value {
            font-size: 20px;
            font-weight: 700;
            color: #0056b3;
        }

        /* Comments Section */
        .comments-section {
            margin-bottom: 20px;
            page-break-inside: avoid;
        }

        .comments-section h3 {
            margin: 0 0 10px 0;
            font-size: 16px;
            color: #2E7D32;
            font-weight: 700;
            border-bottom: 2px solid #e9ecef;
            padding-bottom: 3px;
        }

        .comments-box {
            background: #f8f9fa;
            padding: 12px;
            border-radius: 6px;
            border: 1px solid #dee2e6;
            font-style: italic;
            color: #666;
        }

        /* Signature Section */
        .signature-section {
            margin-top: 40px;
            display: flex;
            justify-content: space-between;
            page-break-inside: avoid;
        }

        .signature-box {
            width: 45%;
            text-align: center;
        }

        .signature-line {
            border-top: 2px solid #2E7D32;
            margin-bottom: 8px;
            height: 1px;
        }

        .signature-box p {
            margin: 3px 0;
            font-size: 12px;
            font-weight: 600;
            color: #2E7D32;
            letter-spacing: 0.05em;
        }

        .signature-box small {
            color: #666;
            font-size: 10px;
        }

        /* Footer */
        .footer-section {
            margin-top: 40px;
            text-align: center;
            border-top: 1px solid #ddd;
            padding-top: 15px;
            page-break-inside: avoid;
        }

        .disclaimer {
            font-size: 12px;
            color: #666;
            font-style: italic;
            margin: 0 0 8px 0;
        }

        .thank-you {
            font-size: 14px;
            font-weight: 600;
            color: #2E7D32;
            margin: 0;
        }

        @media print {
            body {
                background: white;
                box-shadow: none;
                max-width: 100%;
                padding: 20px;
            }
            .no-print {
                display: none;
            }
            .header-section {
                border-bottom: 1px solid #000;
                margin-bottom: 20px;
                padding-bottom: 15px;
            }
            .school-info h1 {
                color: #000;
            }
            .receipt-title h2 {
                color: #000;
            }
            .info-section h3, .payment-section h3, .summary-section h3 {
                color: #000;
                border-bottom: 1px solid #000;
            }
            .amount-display {
                border: 1px solid #000;
                background: white;
            }
            .amount-label, .amount-value {
                color: #000;
            }
            .signature-line {
                border-top: 1px solid #000;
            }
            .signature-box p {
                color: #000;
            }
            .thank-you {
                color: #000;
            }
        }
    </style>
</head>
<body>
    <div class="receipt-container">
        <!-- Header Section -->
        <div class="header-section">
            <div class="school-info">
                <h1><?php echo htmlspecialchars($school_name); ?></h1>
                <p><?php echo htmlspecialchars($school_address); ?></p>
            </div>
            <div class="receipt-title">
                <h2>SALARY PAYMENT RECEIPT</h2>
                <p class="receipt-number">Receipt #: <?php echo htmlspecialchars($receipt['receipt_number'] ?? ''); ?></p>
                <p class="date">Date: <?php echo $report_date; ?></p>
            </div>
        </div>

        <!-- Employee Information Section -->
        <div class="info-section">
            <h3>Employee Information</h3>
            <div class="info-grid">
                <div class="info-row">
                    <label>Employee Name:</label>
                    <span><?php echo htmlspecialchars($receipt['employee_name'] ?? ''); ?></span>
                </div>
                <div class="info-row">
                    <label>Payroll Run:</label>
                    <span><?php echo htmlspecialchars($receipt['payroll_run'] ?? ''); ?></span>
                </div>
                <div class="info-row">
                    <label>Payroll Date:</label>
                    <span><?php echo htmlspecialchars($receipt['payroll_date'] ?? ''); ?></span>
                </div>
                <div class="info-row">
                    <label>Payment Date:</label>
                    <span><?php echo htmlspecialchars($receipt['payment_date'] ?? ''); ?></span>
                </div>
                <div class="info-row">
                    <label>Payment Method:</label>
                    <span><?php echo htmlspecialchars($receipt['payment_method'] ?? ''); ?></span>
                </div>
                <div class="info-row">
                    <label>Processed By:</label>
                    <span><?php echo htmlspecialchars($receipt['processed_by'] ?? 'Unknown'); ?></span>
                </div>
            </div>
        </div>

        <!-- Payment Details Section -->
        <div class="payment-section">
            <h3>Payment Details</h3>
            <div class="amount-display">
                <div class="amount-row">
                    <span class="amount-label">Salary Amount Paid:</span>
                    <span class="amount-value">GHC <?php echo number_format($receipt['amount'] ?? 0, 2); ?></span>
                </div>
            </div>
        </div>

        <!-- Comments Section -->
        <?php if (!empty($receipt['comments'])): ?>
        <div class="comments-section">
            <h3>Payment Comments</h3>
            <div class="comments-box">
                <?php echo htmlspecialchars($receipt['comments']); ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Payment Summary Section -->
        <div class="summary-section">
            <h3>Payment Summary</h3>
            <div class="summary-grid">
                <div class="summary-row">
                    <label>Salary Amount:</label>
                    <span>GHC <?php echo number_format($receipt['amount'] ?? 0, 2); ?></span>
                </div>
                <div class="summary-row status-row">
                    <label>Payment Status:</label>
                    <span class="status"><?php echo htmlspecialchars($receipt['status'] ?? ''); ?></span>
                </div>
            </div>
        </div>

        <!-- Signature Section -->
        <div class="signature-section">
            <div class="signature-box">
                <div class="signature-line"></div>
                <p>Employee Signature</p>
                <small>Date: _______________</small>
            </div>
            <div class="signature-box">
                <div class="signature-line"></div>
                <p>Accountant Signature</p>
                <small>Date: _______________</small>
            </div>
        </div>

        <!-- Footer -->
        <div class="footer-section">
            <p class="disclaimer">This is a computer-generated receipt and is valid without signature.</p>
            <p class="thank-you">Thank you for your service!</p>
        </div>
    </div>

    <script>
        // Auto-print when page loads without user interaction
        window.onload = function() {
            // Small delay to ensure page is fully loaded
            setTimeout(function() {
                window.print();
            }, 100);
        };

        // Close window after printing (for popup windows)
        window.onafterprint = function() {
            // Close the window after printing if it was opened as a popup
            if (window.opener) {
                window.close();
            }
        };
    </script>
</body>
</html>
