<?php
session_start();

$receipt = null;

// Check if receipt data is in session (from payment recording)
if (isset($_SESSION['receipt_data'])) {
    $receipt = $_SESSION['receipt_data'];
    // Clear receipt data after use
    unset($_SESSION['receipt_data']);
}
// Check if fee ID is provided (from fee reports)
elseif (isset($_GET['fee'])) {
    $fee_id = $_GET['fee'];

    // Fetch fee details from database
    require_once '../config.php';

    try {
        $stmt = $pdo->prepare("SELECT sf.*, s.name as student_name, f.name as fee_name, f.amount as fee_amount FROM student_fees sf JOIN students s ON sf.student_id = s.id JOIN fees f ON sf.fee_id = f.id WHERE sf.id = ? AND sf.paid = TRUE");
        $stmt->execute([$fee_id]);
        $fee_data = $stmt->fetch();

        if ($fee_data) {
            // Get accountant name - try multiple sources
            $accountant_name = 'Unknown';

            // First try to get from processed_by field in student_fees table
            if (!empty($fee_data['processed_by'])) {
                $processed_stmt = $pdo->prepare("SELECT CONCAT(first_name, ' ', last_name) as full_name FROM users WHERE id = ?");
                $processed_stmt->execute([$fee_data['processed_by']]);
                $processed_user = $processed_stmt->fetch();
                if ($processed_user && !empty($processed_user['full_name'])) {
                    $accountant_name = $processed_user['full_name'];
                }
            }

            // If not found, try to get from activity logs using receipt number
            if ($accountant_name === 'Unknown' && !empty($fee_data['receipt_number'])) {
                $activity_stmt = $pdo->prepare("
                    SELECT u.username, CONCAT(u.first_name, ' ', u.last_name) as full_name
                    FROM user_activity_logs ual
                    JOIN users u ON ual.user_id = u.id
                    WHERE ual.action = 'fee_payment_recorded'
                    AND ual.description LIKE ?
                    ORDER BY ual.created_at DESC
                    LIMIT 1
                ");
                $activity_stmt->execute(['%Receipt: ' . $fee_data['receipt_number'] . '%']);
                $activity = $activity_stmt->fetch();
                if ($activity) {
                    $accountant_name = $activity['full_name'] ?: $activity['username'];
                }
            }

            // If not found in activity logs, try admins table using current session user_id
            if ($accountant_name === 'Unknown' && isset($_SESSION['user_id'])) {
                $accountant_stmt = $pdo->prepare("SELECT CONCAT(first_name, ' ', last_name) as name FROM admins WHERE user_id = ?");
                $accountant_stmt->execute([$_SESSION['user_id']]);
                $accountant = $accountant_stmt->fetch();
                if ($accountant && $accountant['name']) {
                    $accountant_name = $accountant['name'];
                }
            }

            // If not found, try to get from users table
            if ($accountant_name === 'Unknown' && isset($_SESSION['user_id'])) {
                $user_stmt = $pdo->prepare("SELECT CONCAT(first_name, ' ', last_name) as name FROM users WHERE id = ?");
                $user_stmt->execute([$_SESSION['user_id']]);
                $user = $user_stmt->fetch();
                if ($user && $user['name']) {
                    $accountant_name = $user['name'];
                }
            }

            // If still not found, try username from session
            if ($accountant_name === 'Unknown' && isset($_SESSION['username'])) {
                $accountant_name = $_SESSION['username'];
            }

            // Format receipt data
            $receipt = [
                'receipt_number' => $fee_data['receipt_number'],
                'student_name' => $fee_data['student_name'],
                'fee_name' => $fee_data['fee_name'],
                'amount' => $fee_data['paid_amount'] ?? $fee_data['fee_amount'],
                'total_paid' => $fee_data['paid_amount'] ?? $fee_data['fee_amount'],
                'remaining_balance' => 0, // Since it's paid
                'payment_date' => $fee_data['paid_at'],
                'payment_method' => $fee_data['payment_method'],
                'status' => 'Fully Paid',
                'accountant_name' => $accountant_name
            ];
        }
    } catch (Exception $e) {
        // Handle database error
    }
}

if (!$receipt) {
    echo "No receipt data available.";
    exit;
}

// Fetch school settings
require_once '../config.php';
$school_name = "School Name";
$school_address = "School Address, City, Ghana";

try {
    $stmt = $pdo->prepare("SELECT school_name, address FROM school_settings LIMIT 1");
    $stmt->execute();
    $school_info = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($school_info) {
        $school_name = htmlspecialchars($school_info['school_name']);
        $school_address = htmlspecialchars($school_info['address']);
    }
} catch (Exception $e) {
    // Use default values if database query fails
    $school_name = "School Name";
    $school_address = "School Address, City, Ghana";
}

$report_date = date('Y-m-d');
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Receipt - <?php echo htmlspecialchars($receipt['receipt_number'] ?? ''); ?></title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 0;
            padding: 15px 20px;
            line-height: 1.3;
            color: #222;
            background-color: #fff;
            max-width: 600px;
            margin-left: auto;
            margin-right: auto;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
            font-size: 14px;
        }

        .receipt-container {
            max-width: 100%;
        }

        /* Header Section */
        .header-section {
            display: flex;
            justify-content: center;
            align-items: center;
            flex-direction: column;
            margin-bottom: 30px;
            border-bottom: 3px solid #007BFF;
            padding-bottom: 15px;
            text-align: center;
        }

        .school-info h1 {
            margin: 0 0 8px 0;
            font-size: 22px;
            color: #007BFF;
            font-weight: 700;
            letter-spacing: 1px;
        }

        .school-info p {
            margin: 0;
            font-size: 14px;
            color: #555;
        }

        .receipt-title h2 {
            margin: 10px 0 5px 0;
            font-size: 20px;
            color: #007BFF;
            font-weight: 700;
            letter-spacing: 1px;
        }

        .receipt-number {
            margin: 3px 0;
            font-size: 14px;
            font-weight: 600;
            color: #333;
        }

        .date {
            margin: 3px 0;
            font-size: 12px;
            color: #666;
        }

        /* Information Sections */
        .info-section, .payment-section, .summary-section {
            margin-bottom: 20px;
            page-break-inside: avoid;
        }

        .info-section h3, .payment-section h3, .summary-section h3 {
            margin: 0 0 10px 0;
            font-size: 16px;
            color: #007BFF;
            font-weight: 700;
            border-bottom: 2px solid #e9ecef;
            padding-bottom: 3px;
        }

        .info-grid, .summary-grid {
            display: grid;
            grid-template-columns: 1fr;
            gap: 8px;
        }

        .info-row, .summary-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 0;
            border-bottom: 1px solid #f0f0f0;
        }

        .info-row:last-child, .summary-row:last-child {
            border-bottom: none;
        }

        .info-row label, .summary-row label {
            font-weight: 600;
            color: #555;
            min-width: 140px;
        }

        .info-row span, .summary-row span {
            font-weight: 500;
            color: #333;
            text-align: right;
        }

        .status-row span.status {
            font-weight: 700;
            padding: 4px 12px;
            border-radius: 4px;
            background: #28a745;
            color: white;
        }

        /* Payment Details */
        .amount-display {
            background: #f8f9fa;
            border: 2px solid #007BFF;
            border-radius: 6px;
            padding: 15px;
            text-align: center;
        }

        .amount-row {
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 15px;
        }

        .amount-label {
            font-size: 16px;
            font-weight: 600;
            color: #007BFF;
        }

        .amount-value {
            font-size: 20px;
            font-weight: 700;
            color: #0056b3;
        }

        /* Signature Section */
        .signature-section {
            margin-top: 40px;
            display: flex;
            justify-content: space-between;
            page-break-inside: avoid;
        }

        .signature-box {
            width: 45%;
            text-align: center;
        }

        .signature-line {
            border-top: 2px solid #007BFF;
            margin-bottom: 8px;
            height: 1px;
        }

        .signature-box p {
            margin: 3px 0;
            font-size: 12px;
            font-weight: 600;
            color: #007BFF;
            letter-spacing: 0.05em;
        }

        .signature-box small {
            color: #666;
            font-size: 10px;
        }

        /* Footer */
        .footer-section {
            margin-top: 40px;
            text-align: center;
            border-top: 1px solid #ddd;
            padding-top: 15px;
            page-break-inside: avoid;
        }

        .disclaimer {
            font-size: 12px;
            color: #666;
            font-style: italic;
            margin: 0 0 8px 0;
        }

        .thank-you {
            font-size: 14px;
            font-weight: 600;
            color: #007BFF;
            margin: 0;
        }

        @media print {
            body {
                background: white;
                box-shadow: none;
                max-width: 100%;
                padding: 20px;
            }
            .no-print {
                display: none;
            }
            .header-section {
                border-bottom: 1px solid #000;
                margin-bottom: 20px;
                padding-bottom: 15px;
            }
            .school-info h1 {
                color: #000;
            }
            .receipt-title h2 {
                color: #000;
            }
            .info-section h3, .payment-section h3, .summary-section h3 {
                color: #000;
                border-bottom: 1px solid #000;
            }
            .amount-display {
                border: 1px solid #000;
                background: white;
            }
            .amount-label, .amount-value {
                color: #000;
            }
            .signature-line {
                border-top: 1px solid #000;
            }
            .signature-box p {
                color: #000;
            }
            .thank-you {
                color: #000;
            }
        }
    </style>
</head>
<body>
    <div class="receipt-container">
        <!-- Header Section -->
        <div class="header-section">
            <div class="school-info">
                <h1><?php echo htmlspecialchars($school_name); ?></h1>
                <p><?php echo htmlspecialchars($school_address); ?></p>
            </div>
            <div class="receipt-title">
                <h2>PAYMENT RECEIPT</h2>
                <p class="receipt-number">Receipt #: <?php echo htmlspecialchars($receipt['receipt_number'] ?? ''); ?></p>
                <p class="date">Date: <?php echo $report_date; ?></p>
            </div>
        </div>

        <!-- Student Information Section -->
        <div class="info-section">
            <h3>Student Information</h3>
            <div class="info-grid">
                <div class="info-row">
                    <label>Student Name:</label>
                    <span><?php echo htmlspecialchars($receipt['student_name'] ?? ''); ?></span>
                </div>
                <div class="info-row">
                    <label>Fee Description:</label>
                    <span><?php echo htmlspecialchars($receipt['fee_name'] ?? ''); ?></span>
                </div>
                <div class="info-row">
                    <label>Payment Date:</label>
                    <span><?php echo htmlspecialchars($receipt['payment_date'] ?? ''); ?></span>
                </div>
                <div class="info-row">
                    <label>Payment Method:</label>
                    <span><?php echo htmlspecialchars($receipt['payment_method'] ?? ''); ?></span>
                </div>
                <div class="info-row">
                    <label>Processed By:</label>
                    <span><?php echo htmlspecialchars($receipt['accountant_name'] ?? 'Unknown'); ?></span>
                </div>
            </div>
        </div>

        <!-- Payment Details Section -->
        <div class="payment-section">
            <h3>Payment Details</h3>
            <div class="amount-display">
                <div class="amount-row">
                    <span class="amount-label">Amount Paid:</span>
                    <span class="amount-value">GHC <?php echo number_format($receipt['amount'] ?? 0, 2); ?></span>
                </div>
            </div>
        </div>

        <!-- Financial Summary Section -->
        <div class="summary-section">
            <h3>Financial Summary</h3>
            <div class="summary-grid">
                <div class="summary-row">
                    <label>Total Fee Amount:</label>
                    <span>GHC <?php echo number_format(($receipt['total_paid'] ?? 0) + ($receipt['remaining_balance'] ?? 0), 2); ?></span>
                </div>
                <div class="summary-row">
                    <label>Total Paid to Date:</label>
                    <span>GHC <?php echo number_format($receipt['total_paid'] ?? 0, 2); ?></span>
                </div>
                <div class="summary-row">
                    <label>Outstanding Balance:</label>
                    <span>GHC <?php echo number_format($receipt['remaining_balance'] ?? 0, 2); ?></span>
                </div>
                <div class="summary-row status-row">
                    <label>Payment Status:</label>
                    <span class="status"><?php echo htmlspecialchars($receipt['status'] ?? ''); ?></span>
                </div>
            </div>
        </div>

        <!-- Footer -->
        <div class="footer-section">
            <p class="disclaimer">This is a computer-generated receipt and is valid without signature.</p>
            <p class="thank-you">Thank you for your payment!</p>
        </div>
    </div>

    <script>
        // Auto-print when page loads without user interaction
        window.onload = function() {
            // Small delay to ensure page is fully loaded
            setTimeout(function() {
                window.print();
            }, 100);
        };

        // Close window after printing (for popup windows)
        window.onafterprint = function() {
            // Close the window after printing if it was opened as a popup
            if (window.opener) {
                window.close();
            }
        };
    </script>
</body>
</html>
