<?php
session_start();

// Check permissions - only admins and accounts users can manage purchase order receiving
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['receive_po'])) {
        // Receive purchase order
        $po_id = intval($_POST['po_id']);
        $received_items = json_decode($_POST['received_items'], true);
        $notes = trim($_POST['notes'] ?? '');

        if (empty($po_id) || empty($received_items)) {
            $error = "Purchase order ID and received items are required.";
        } else {
            try {
                // Start transaction
                $pdo->beginTransaction();

                // Get PO details
                $stmt = $pdo->prepare("
                    SELECT po.*, s.name AS supplier_name
                    FROM purchase_orders po
                    JOIN suppliers s ON po.supplier_id = s.id
                    WHERE po.id = ? AND po.status = 'approved'
                ");
                $stmt->execute([$po_id]);
                $po = $stmt->fetch();

                if (!$po) {
                    throw new Exception("Purchase order not found or not approved.");
                }

                // Update PO status to received
                $stmt = $pdo->prepare("
                    UPDATE purchase_orders
                    SET status = 'received', updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$po_id]);

                // Process received items
                foreach ($received_items as $item) {
                    $item_id = intval($item['item_id']);
                    $quantity_received = floatval($item['quantity_received']);
                    $condition = trim($item['condition']);
                    $location = trim($item['location'] ?? '');

                    if ($quantity_received > 0) {
                        // Insert into purchase_order_receipts table
                        $stmt = $pdo->prepare("
                            INSERT INTO purchase_order_receipts
                            (purchase_order_id, purchase_order_item_id, quantity_received, condition_status, location, received_by, received_at, notes)
                            VALUES (?, ?, ?, ?, ?, ?, NOW(), ?)
                        ");
                        $stmt->execute([
                            $po_id,
                            $item_id,
                            $quantity_received,
                            $condition,
                            $location,
                            $_SESSION['user_id'],
                            $notes
                        ]);

                        // Update inventory if applicable (you may need to adjust this based on your inventory system)
                        // For now, we'll just log the receipt
                    }
                }

                // Log activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'po_received',
                    "Received purchase order: {$po['po_number']} from {$po['supplier_name']}"
                );

                $pdo->commit();
                $message = "Purchase order {$po['po_number']} has been successfully received!";

            } catch (Exception $e) {
                $pdo->rollBack();
                $error = "Error receiving purchase order: " . $e->getMessage();
            }
        }
    }
}

// Get approved purchase orders ready for receiving
try {
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';

    $query = "
        SELECT po.*, s.name AS supplier_name,
               u.username AS created_by_name,
               (
                   SELECT COUNT(*) FROM purchase_order_items poi WHERE poi.purchase_order_id = po.id
               ) as item_count,
               (
                   SELECT COALESCE(SUM(por.quantity_received), 0)
                   FROM purchase_order_receipts por
                   JOIN purchase_order_items poi ON por.purchase_order_item_id = poi.id
                   WHERE poi.purchase_order_id = po.id
               ) as total_received
        FROM purchase_orders po
        JOIN suppliers s ON po.supplier_id = s.id
        LEFT JOIN users u ON po.created_by = u.id
        WHERE po.status = 'approved'
    ";

    $params = [];

    if (!empty($search)) {
        $query .= " AND (po.po_number LIKE ? OR s.name LIKE ?)";
        $params[] = "%$search%";
        $params[] = "%$search%";
    }

    $query .= " ORDER BY po.created_at DESC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $approved_pos = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading approved purchase orders: " . $e->getMessage();
    $approved_pos = [];
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .receiving-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .receiving-header {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .receiving-content {
        display: grid;
        grid-template-columns: 1fr;
        gap: 30px;
    }

    .receiving-section {
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .section-title {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
        font-size: 1.5em;
    }

    .filters-section {
        display: flex;
        gap: 20px;
        margin-bottom: 20px;
        align-items: center;
    }

    .filter-group {
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .filter-group label {
        font-weight: 600;
        color: #495057;
    }

    .filter-group input {
        padding: 8px 12px;
        border: 1px solid #ced4da;
        border-radius: 5px;
        font-size: 14px;
    }

    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        margin-right: 10px;
        transition: all 0.3s ease;
    }

    .btn-primary {
        background: #007bff;
        color: white;
    }

    .btn-primary:hover {
        background: #0056b3;
        transform: translateY(-2px);
    }

    .btn-success {
        background: #28a745;
        color: white;
    }

    .btn-success:hover {
        background: #1e7e34;
        transform: translateY(-2px);
    }

    .message {
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }

    .success-message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .po-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .po-table th,
    .po-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .po-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #495057;
    }

    .po-number {
        font-weight: bold;
        color: #007bff;
    }

    .supplier-name {
        font-weight: 600;
        color: #28a745;
    }

    .status-badge {
        display: inline-block;
        padding: 4px 12px;
        border-radius: 20px;
        font-size: 0.85em;
        font-weight: 600;
        text-transform: uppercase;
    }

    .status-approved {
        background: #d1ecf1;
        color: #0c5460;
    }

    .amount {
        font-weight: 600;
        color: #007bff;
    }

    .actions-column {
        width: 150px;
    }

    .progress-bar {
        width: 100%;
        height: 8px;
        background: #e9ecef;
        border-radius: 4px;
        overflow: hidden;
    }

    .progress-fill {
        height: 100%;
        background: linear-gradient(90deg, #28a745, #20c997);
        transition: width 0.3s ease;
    }

    @media (max-width: 768px) {
        .filters-section {
            flex-direction: column;
            align-items: stretch;
        }

        .po-table {
            font-size: 0.9em;
        }

        .actions-column {
            width: 120px;
        }
    }
</style>

<div class="receiving-container">
    <div class="receiving-header">
        <h1>📦 Purchase Order Receiving</h1>
        <p>Receive and process approved purchase orders from suppliers</p>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="receiving-content">
        <div class="receiving-section">
            <div class="section-title">📋 Approved Purchase Orders Ready for Receiving</div>

            <div class="filters-section">
                <div class="filter-group">
                    <label for="search">Search:</label>
                    <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($search ?? ''); ?>" placeholder="PO number or supplier...">
                </div>
                <button class="btn btn-primary" onclick="applyFilters()">🔍 Filter</button>
            </div>

            <?php if (empty($approved_pos)): ?>
                <div style="text-align: center; padding: 60px 20px; color: #6c757d;">
                    <div style="font-size: 4em; margin-bottom: 15px;">📦</div>
                    <h3>No Purchase Orders Ready for Receiving</h3>
                    <p>All approved purchase orders have been processed. Check back later for new approved orders.</p>
                </div>
            <?php else: ?>
                <table class="po-table">
                    <thead>
                        <tr>
                            <th>PO Number</th>
                            <th>Supplier</th>
                            <th>Items</th>
                            <th>Total Amount</th>
                            <th>Expected Delivery</th>
                            <th>Receiving Progress</th>
                            <th class="actions-column">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($approved_pos as $po): ?>
                            <?php
                            $progress_percentage = $po['item_count'] > 0 ? min(100, ($po['total_received'] / $po['item_count']) * 100) : 0;
                            ?>
                            <tr>
                                <td>
                                    <div class="po-number"><?php echo htmlspecialchars($po['po_number']); ?></div>
                                </td>
                                <td>
                                    <div class="supplier-name"><?php echo htmlspecialchars($po['supplier_name']); ?></div>
                                </td>
                                <td><?php echo $po['item_count']; ?> items</td>
                                <td class="amount">GHC <?php echo number_format($po['total_amount'], 2); ?></td>
                                <td><?php echo date('M j, Y', strtotime($po['expected_delivery_date'])); ?></td>
                                <td>
                                    <div style="display: flex; align-items: center; gap: 10px;">
                                        <div class="progress-bar">
                                            <div class="progress-fill" style="width: <?php echo $progress_percentage; ?>%"></div>
                                        </div>
                                        <span style="font-size: 0.9em; color: #6c757d;"><?php echo round($progress_percentage); ?>%</span>
                                    </div>
                                </td>
                                <td class="actions-column">
                                    <button class="btn btn-success btn-sm" onclick="receivePO(<?php echo $po['id']; ?>)">📦 Receive</button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Receive PO Modal -->
<div id="receiveModal" class="modal">
    <div class="modal-content modern-modal">
        <div class="modal-header">
            <div class="modal-title-section">
                <div class="modal-icon">📦</div>
                <div>
                    <h2 class="modal-title">Receive Purchase Order</h2>
                    <p class="modal-subtitle">Record the receipt of ordered items</p>
                </div>
            </div>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>

        <form method="post" id="receivePOForm">
            <input type="hidden" name="po_id" id="receive_po_id">

            <!-- PO Details Section -->
            <div class="form-section">
                <div class="section-header">
                    <h3 class="section-title">📋 Purchase Order Details</h3>
                    <div class="section-divider"></div>
                </div>

                <div id="poDetails">
                    <!-- PO details will be loaded here -->
                </div>
            </div>

            <!-- Items Receiving Section -->
            <div class="form-section">
                <div class="section-header">
                    <h3 class="section-title">📦 Item Receipt Details</h3>
                    <div class="section-divider"></div>
                </div>

                <div class="items-receiving-container">
                    <div id="itemsReceivingList">
                        <!-- Items will be loaded here -->
                    </div>
                </div>
            </div>

            <!-- Notes Section -->
            <div class="form-section">
                <div class="section-header">
                    <h3 class="section-title">📝 Additional Notes</h3>
                    <div class="section-divider"></div>
                </div>

                <div class="form-group">
                    <label for="notes" class="form-label">
                        <i class="fas fa-sticky-note"></i> Notes (Optional)
                    </label>
                    <div class="input-wrapper">
                        <textarea name="notes" id="notes" class="form-input" rows="3" placeholder="Add any notes about the receipt..."></textarea>
                        <div class="input-icon">📝</div>
                    </div>
                </div>
            </div>

            <!-- Form Actions -->
            <div class="form-actions">
                <button type="button" class="btn btn-secondary" onclick="closeModal()">
                    <i class="fas fa-times"></i>
                    Cancel
                </button>
                <button type="submit" name="receive_po" class="btn btn-success" id="submitReceiveBtn">
                    <i class="fas fa-check"></i>
                    Complete Receipt
                </button>
            </div>

            <input type="hidden" name="received_items" id="received_items">
        </form>
    </div>
</div>

<style>
/* Modern Modal Styles */
.modern-modal {
    max-width: 1000px;
    width: 95%;
    max-height: 90vh;
    overflow-y: auto;
}

.modal-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 25px 30px;
    border-radius: 15px 15px 0 0;
    margin: -30px -30px 30px -30px;
}

.modal-title-section {
    display: flex;
    align-items: center;
    gap: 15px;
}

.modal-icon {
    font-size: 2.5em;
    background: rgba(255,255,255,0.2);
    padding: 10px;
    border-radius: 12px;
}

.modal-title {
    margin: 0;
    font-size: 1.8em;
    font-weight: 700;
}

.modal-subtitle {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1em;
}

.close {
    font-size: 28px;
    cursor: pointer;
    color: rgba(255,255,255,0.8);
    transition: color 0.3s ease;
}

.close:hover {
    color: white;
}

/* Form Sections */
.form-section {
    margin-bottom: 30px;
    background: #f8f9fa;
    border-radius: 12px;
    padding: 25px;
    border: 1px solid #e9ecef;
}

.section-header {
    display: flex;
    align-items: center;
    margin-bottom: 20px;
}

.section-title {
    margin: 0;
    font-size: 1.3em;
    font-weight: 600;
    color: #2c3e50;
    display: flex;
    align-items: center;
    gap: 8px;
}

.section-divider {
    flex: 1;
    height: 2px;
    background: linear-gradient(90deg, #667eea, #764ba2);
    margin-left: 15px;
    border-radius: 1px;
}

/* PO Details */
.po-details-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 20px;
}

.detail-item {
    background: white;
    padding: 15px;
    border-radius: 8px;
    border: 1px solid #e9ecef;
}

.detail-label {
    font-weight: 600;
    color: #495057;
    font-size: 0.9em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-bottom: 5px;
}

.detail-value {
    font-size: 1.1em;
    color: #2c3e50;
    font-weight: 500;
}

/* Items Receiving */
.items-receiving-container {
    background: white;
    border-radius: 10px;
    border: 1px solid #e9ecef;
    overflow: hidden;
}

.receiving-item {
    padding: 20px;
    border-bottom: 1px solid #f0f0f0;
}

.receiving-item:last-child {
    border-bottom: none;
}

.item-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
}

.item-title {
    font-weight: 600;
    color: #2c3e50;
    font-size: 1.1em;
}

.item-quantity {
    color: #6c757d;
    font-size: 0.9em;
}

.receiving-form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr 1fr 1fr;
    gap: 15px;
    align-items: end;
}

.form-group {
    margin-bottom: 0;
}

.form-label {
    display: block;
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
    font-size: 0.9em;
}

.input-wrapper {
    position: relative;
}

.form-input {
    width: 100%;
    padding: 10px 12px;
    border: 2px solid #e9ecef;
    border-radius: 6px;
    font-size: 14px;
    transition: border-color 0.3s ease;
    background: white;
}

.form-input:focus {
    border-color: #667eea;
    outline: none;
}

textarea.form-input {
    resize: vertical;
    min-height: 80px;
}

.input-icon {
    position: absolute;
    right: 10px;
    top: 50%;
    transform: translateY(-50%);
    font-size: 1em;
    color: #6c757d;
    pointer-events: none;
}

/* Form Actions */
.form-actions {
    display: flex;
    justify-content: flex-end;
    gap: 15px;
    padding-top: 30px;
    border-top: 1px solid #e9ecef;
}

.btn {
    padding: 12px 25px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    transition: all 0.3s ease;
    min-width: 140px;
    justify-content: center;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-2px);
}

.btn-success {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(40, 167, 69, 0.4);
}

.btn-success:disabled {
    background: #6c757d;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

/* Responsive Design */
@media (max-width: 768px) {
    .modern-modal {
        width: 98%;
        margin: 1%;
    }

    .modal-header {
        padding: 20px;
    }

    .modal-title {
        font-size: 1.5em;
    }

    .form-section {
        padding: 20px;
    }

    .receiving-form-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }

    .po-details-grid {
        grid-template-columns: 1fr;
    }

    .form-actions {
        flex-direction: column;
    }

    .btn {
        width: 100%;
    }
}
</style>

<script>
function receivePO(poId) {
    // Load PO details and items
    fetch(`api/get_po_details.php?po_id=${poId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayReceiveModal(data.po, data.items);
            } else {
                alert('Error loading purchase order details: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading purchase order details');
        });
}

function displayReceiveModal(po, items) {
    document.getElementById('receive_po_id').value = po.id;

    // Display PO details
    const poDetails = document.getElementById('poDetails');
    poDetails.innerHTML = `
        <div class="po-details-grid">
            <div class="detail-item">
                <div class="detail-label">PO Number</div>
                <div class="detail-value">${po.po_number}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Supplier</div>
                <div class="detail-value">${po.supplier_name}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Total Amount</div>
                <div class="detail-value">GHC ${parseFloat(po.total_amount).toFixed(2)}</div>
            </div>
            <div class="detail-item">
                <div class="detail-label">Expected Delivery</div>
                <div class="detail-value">${new Date(po.expected_delivery_date).toLocaleDateString()}</div>
            </div>
        </div>
    `;

    // Display items for receiving
    const itemsList = document.getElementById('itemsReceivingList');
    itemsList.innerHTML = items.map(item => `
        <div class="receiving-item">
            <div class="item-header">
                <div class="item-title">${item.description}</div>
                <div class="item-quantity">Ordered: ${item.quantity_ordered} ${item.unit || ''}</div>
            </div>
            <div class="receiving-form-grid">
                <div class="form-group">
                    <label class="form-label">Received Qty</label>
                    <input type="number" class="form-input item-received-qty"
                           data-item-id="${item.id}"
                           min="0" max="${item.quantity_ordered}"
                           step="0.01" placeholder="0.00" required>
                </div>
                <div class="form-group">
                    <label class="form-label">Condition</label>
                    <select class="form-input item-condition" data-item-id="${item.id}" required>
                        <option value="">Select condition</option>
                        <option value="excellent">Excellent</option>
                        <option value="good">Good</option>
                        <option value="fair">Fair</option>
                        <option value="poor">Poor</option>
                        <option value="damaged">Damaged</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-label">Location</label>
                    <input type="text" class="form-input item-location"
                           data-item-id="${item.id}"
                           placeholder="Storage location">
                </div>
                <div class="form-group">
                    <label class="form-label">Unit Price</label>
                    <input type="number" class="form-input"
                           value="${parseFloat(item.unit_price).toFixed(2)}"
                           readonly step="0.01">
                </div>
            </div>
        </div>
    `).join('');

    document.getElementById('receiveModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('receiveModal').style.display = 'none';
    document.getElementById('receivePOForm').reset();
}

function applyFilters() {
    const search = document.getElementById('search').value;
    let url = 'po_receiving.php?';
    if (search) url += `search=${encodeURIComponent(search)}&`;
    window.location.href = url.slice(0, -1);
}

// Form validation and submission
document.getElementById('receivePOForm').addEventListener('submit', function(e) {
    const receivedItems = [];
    let hasValidItems = false;

    document.querySelectorAll('.receiving-item').forEach(item => {
        const itemId = item.querySelector('.item-received-qty').dataset.itemId;
        const quantityReceived = parseFloat(item.querySelector('.item-received-qty').value) || 0;
        const condition = item.querySelector('.item-condition').value;
        const location = item.querySelector('.item-location').value;

        if (quantityReceived > 0 && condition) {
            receivedItems.push({
                item_id: itemId,
                quantity_received: quantityReceived,
                condition: condition,
                location: location
            });
            hasValidItems = true;
        }
    });

    if (!hasValidItems) {
        e.preventDefault();
        alert('Please enter at least one item with quantity received and condition.');
        return;
    }

    document.getElementById('received_items').value = JSON.stringify(receivedItems);
});

// Close modal when clicking outside
window.onclick = function(event) {
    if (event.target.className === 'modal') {
        event.target.style.display = 'none';
    }
}

// Initialize when page loads
document.addEventListener('DOMContentLoaded', function() {
    // Any initialization code can go here
});
</script>

<?php include '../includes/admin_footer.php'; ?>
