<?php
require_once '../config.php';
require_once 'accounting_functions.php';

if (!function_exists('setup_logger')) {
    function setup_logger() {
        return null; // Dummy logger for testing
    }
}

/**
 * Create journal entry for PO payment (cash basis)
 * Debit: Accounts Payable, Credit: Cash/Bank Account
 */
function createPOPaymentEntry($pdo, $poId, $amount, $paymentMethod, $paymentDate, $paidBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get PO details
        $stmt = $pdo->prepare("
            SELECT po.*, s.name AS supplier_name
            FROM purchase_orders po
            JOIN suppliers s ON po.supplier_id = s.id
            WHERE po.id = ?
        ");
        $stmt->execute([$poId]);
        $po = $stmt->fetch();

        if (!$po) {
            throw new Exception('Purchase order not found');
        }

        // Get Accounts Payable account
        $payableAccountId = getChartAccountId($pdo, 'Accounts Payable');
        if (!$payableAccountId) {
            throw new Exception('Accounts Payable account not found');
        }

        // Determine cash/bank account based on payment method
        $cashAccountName = '';
        switch ($paymentMethod) {
            case 'cash':
                $cashAccountName = 'Petty Cash';
                break;
            default:
                $cashAccountName = 'Main School Bank Account';
        }

        $cashAccountId = getChartAccountId($pdo, $cashAccountName);
        if (!$cashAccountId) {
            throw new Exception("{$cashAccountName} account not found");
        }

        $entryNumber = generateEntryNumber($pdo, $paymentDate);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'po_payment', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $paymentDate,
            "PO Payment: {$po['po_number']} to {$po['supplier_name']} via {$paymentMethod}",
            $poId,
            $paidBy
        ]);
        $entryId = $pdo->lastInsertId();

        $lineNumber = 1;

        // Debit: Accounts Payable (reduces liability)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $entryId,
            $payableAccountId,
            $amount,
            $lineNumber++,
            'Accounts payable decrease (PO payment)'
        ]);

        // Update account balance
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
        $stmt->execute([$amount, $payableAccountId]); // Debit decreases liability

        // Credit: Cash/Bank Account (reduces cash)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $entryId,
            $cashAccountId,
            $amount,
            $lineNumber,
            "Payment to supplier: {$po['supplier_name']}"
        ]);

        // Update account balance
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
        $stmt->execute([$amount, $cashAccountId]); // Credit decreases cash

        if ($manageTransaction) {
            $pdo->commit();
        }

        // Update PO status based on total payments
        $stmt = $pdo->prepare("SELECT SUM(amount) as total_paid FROM purchase_order_payments WHERE purchase_order_id = ?");
        $stmt->execute([$poId]);
        $totalPaid = $stmt->fetch()['total_paid'] ?? 0;

        $poTotal = $po['total_amount'];
        if ($totalPaid >= $poTotal) {
            $status = 'paid';
        } elseif ($totalPaid > 0) {
            $status = 'partially_paid';
        } else {
            $status = 'pending';
        }

        $stmt = $pdo->prepare("UPDATE purchase_orders SET status = ? WHERE id = ?");
        $stmt->execute([$status, $poId]);

        $logger = setup_logger();
        $logger->info("PO payment journal entry created: {$entryNumber} for PO ID {$poId}");

        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}
?>
