<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';

// Check permissions - only admins and accounts users can manage suppliers
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_supplier'])) {
        // Add new supplier
        $supplier_name = trim($_POST['supplier_name']);
        $contact_person = trim($_POST['contact_person']);
        $email = trim($_POST['email']);
        $phone = trim($_POST['phone']);
        $address = trim($_POST['address']);

        if (empty($supplier_name)) {
            $error = "Supplier name is required.";
        } elseif (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = "Please enter a valid email address.";
        } else {
            try {
        // Check if supplier name already exists
                $stmt = $pdo->prepare("SELECT id FROM suppliers WHERE name = ?");
                $stmt->execute([$supplier_name]);
                if ($stmt->fetch()) {
                    $error = "A supplier with this name already exists.";
                } else {
                    $stmt = $pdo->prepare("
                        INSERT INTO suppliers (name, contact_person, email, phone, address, status)
                        VALUES (?, ?, ?, ?, ?, 'active')
                    ");
                    $stmt->execute([$supplier_name, $contact_person, $email, $phone, $address]);

                    // Log activity
                    AccountantActivityLogger::logActivity(
                        $_SESSION['user_id'],
                        'supplier_created',
                        "Created new supplier: $supplier_name"
                    );

                    $message = "Supplier added successfully!";
                }
            } catch (PDOException $e) {
                $error = "Error adding supplier: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['edit_supplier'])) {
        // Edit existing supplier
        $id = $_POST['supplier_id'];
        $supplier_name = trim($_POST['supplier_name']);
        $contact_person = trim($_POST['contact_person']);
        $email = trim($_POST['email']);
        $phone = trim($_POST['phone']);
        $address = trim($_POST['address']);

        if (empty($supplier_name)) {
            $error = "Supplier name is required.";
        } elseif (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = "Please enter a valid email address.";
        } else {
            try {
                // Check if another supplier with this name exists
                $stmt = $pdo->prepare("SELECT id FROM suppliers WHERE name = ? AND id != ?");
                $stmt->execute([$supplier_name, $id]);
                if ($stmt->fetch()) {
                    $error = "Another supplier with this name already exists.";
                } else {
                    $stmt = $pdo->prepare("
                        UPDATE suppliers
                        SET name = ?, contact_person = ?, email = ?, phone = ?, address = ?, updated_at = NOW()
                        WHERE id = ?
                    ");
                    $stmt->execute([$supplier_name, $contact_person, $email, $phone, $address, $id]);

                    // Log activity
                    AccountantActivityLogger::logActivity(
                        $_SESSION['user_id'],
                        'supplier_updated',
                        "Updated supplier: $supplier_name"
                    );

                    $message = "Supplier updated successfully!";
                }
            } catch (PDOException $e) {
                $error = "Error updating supplier: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['toggle_active'])) {
        // Toggle active status
        $id = $_POST['supplier_id'];

        try {
            $stmt = $pdo->prepare("
                UPDATE suppliers
                SET status = CASE WHEN status = 'active' THEN 'inactive' ELSE 'active' END, updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$id]);

            if ($stmt->rowCount() > 0) {
                $status = $_POST['current_status'] == 'active' ? 'deactivated' : 'activated';
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'supplier_status_changed',
                    "Supplier ID $id $status"
                );

                $message = "Supplier status updated successfully!";
            } else {
                $error = "Supplier not found.";
            }
        } catch (PDOException $e) {
            $error = "Error updating supplier status: " . $e->getMessage();
        }
    }
}

// Get all suppliers for display
try {
    $stmt = $pdo->query("
        SELECT s.*,
               (
                   SELECT COUNT(*) FROM purchase_orders po WHERE po.supplier_id = s.id
               ) as order_count
        FROM suppliers s
        ORDER BY s.name ASC
    ");
    $suppliers = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading suppliers: " . $e->getMessage();
    $suppliers = [];
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .settings-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .settings-header {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .settings-content {
        display: grid;
        grid-template-columns: 1fr 2fr;
        gap: 30px;
    }

    .add-supplier-section,
    .suppliers-list-section {
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .section-title {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
    }

    .form-group {
        margin-bottom: 20px;
    }

    .form-group label {
        display: block;
        font-weight: 600;
        margin-bottom: 5px;
        color: #495057;
    }

    .form-group input,
    .form-group textarea {
        width: 100%;
        padding: 10px 12px;
        border: 1px solid #ced4da;
        border-radius: 5px;
        font-size: 14px;
    }

    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }

    .form-row {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
    }

    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        margin-right: 10px;
    }

    .btn-primary {
        background: #007bff;
        color: white;
    }

    .btn-success {
        background: #28a745;
        color: white;
    }

    .btn-danger {
        background: #dc3545;
        color: white;
    }

    .btn-secondary {
        background: #6c757d;
        color: white;
    }

    .btn-warning {
        background: #ffc107;
        color: #212529;
    }

    .message {
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }

    .success-message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .suppliers-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .suppliers-table th,
    .suppliers-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .suppliers-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #495057;
    }

    .supplier-name {
        font-weight: bold;
        color: #007bff;
    }

    .contact-info {
        font-size: 0.9em;
        color: #6c757d;
    }

    .status-badge {
        display: inline-block;
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: 600;
    }

    .status-active {
        background: #d4edda;
        color: #155724;
    }

    .status-inactive {
        background: #f8d7da;
        color: #721c24;
    }

    .actions-column {
        width: 200px;
    }

    .modal {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        z-index: 1000;
    }

    .modal-content {
        background: white;
        margin: 5% auto;
        padding: 30px;
        border-radius: 10px;
        width: 90%;
        max-width: 600px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        max-height: 90vh;
        overflow-y: auto;
    }

    .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
        padding-bottom: 10px;
        border-bottom: 1px solid #e9ecef;
    }

    .modal-title {
        margin: 0;
        color: #2c3e50;
    }

    .close {
        font-size: 24px;
        cursor: pointer;
        color: #6c757d;
    }

    @media (max-width: 768px) {
        .settings-content {
            grid-template-columns: 1fr;
        }

        .form-row {
            grid-template-columns: 1fr;
        }
    }
</style>

<div class="settings-container">
    <div class="settings-header">
        <h1>🏢 Supplier Management</h1>
        <p>Manage your school's suppliers</p>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="settings-content">
        <div class="add-supplier-section">
            <h2 class="section-title">➕ Add New Supplier</h2>
            <form method="post">
                <div class="form-group">
                    <label for="supplier_name">Supplier Name *</label>
                    <input type="text" name="supplier_name" id="supplier_name" required placeholder="e.g., ABC Supplies Ltd">
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="contact_person">Contact Person</label>
                        <input type="text" name="contact_person" id="contact_person" placeholder="e.g., John Doe">
                    </div>

                    <div class="form-group">
                        <label for="email">Email</label>
                        <input type="email" name="email" id="email" placeholder="supplier@example.com">
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="phone">Phone</label>
                        <input type="text" name="phone" id="phone" placeholder="+233 XX XXX XXXX">
                    </div>

                    <div class="form-group">
                        <label for="address">Address</label>
                        <input type="text" name="address" id="address" placeholder="Supplier address">
                    </div>
                </div>

                <button type="submit" name="add_supplier" class="btn btn-primary">Add Supplier</button>
            </form>
        </div>

        <div class="suppliers-list-section">
            <h2 class="section-title">📋 All Suppliers</h2>
            <table class="suppliers-table">
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Contact</th>
                        <th>Status</th>
                        <th>Orders</th>
                        <th class="actions-column">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($suppliers as $supplier): ?>
                        <tr>
                            <td>
                                <div class="supplier-name"><?php echo htmlspecialchars($supplier['name']); ?></div>
                                <?php if ($supplier['address']): ?>
                                    <div class="contact-info"><?php echo htmlspecialchars($supplier['address']); ?></div>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($supplier['contact_person']): ?>
                                    <div><?php echo htmlspecialchars($supplier['contact_person']); ?></div>
                                <?php endif; ?>
                                <?php if ($supplier['email']): ?>
                                    <div class="contact-info"><?php echo htmlspecialchars($supplier['email']); ?></div>
                                <?php endif; ?>
                                <?php if ($supplier['phone']): ?>
                                    <div class="contact-info"><?php echo htmlspecialchars($supplier['phone']); ?></div>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="status-badge <?php echo $supplier['status'] == 'active' ? 'status-active' : 'status-inactive'; ?>">
                                    <?php echo $supplier['status'] == 'active' ? 'Active' : 'Inactive'; ?>
                                </span>
                            </td>
                            <td><?php echo $supplier['order_count']; ?> orders</td>
                            <td class="actions-column">
                                <button class="btn btn-secondary btn-sm" onclick="editSupplier(<?php echo $supplier['id']; ?>, '<?php echo addslashes($supplier['name']); ?>', '<?php echo addslashes($supplier['contact_person'] ?? ''); ?>', '<?php echo addslashes($supplier['email'] ?? ''); ?>', '<?php echo addslashes($supplier['phone'] ?? ''); ?>', '<?php echo addslashes($supplier['address'] ?? ''); ?>')">Edit</button>
                                <button class="btn <?php echo $supplier['status'] == 'active' ? 'btn-warning' : 'btn-success'; ?> btn-sm" onclick="toggleStatus(<?php echo $supplier['id']; ?>, <?php echo $supplier['status'] == 'active' ? 'true' : 'false'; ?>)">
                                    <?php echo $supplier['status'] == 'active' ? 'Deactivate' : 'Activate'; ?>
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Edit Supplier Modal -->
<div id="editModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Edit Supplier</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <form method="post">
            <input type="hidden" name="supplier_id" id="edit_supplier_id">
            <div class="form-group">
                <label for="edit_supplier_name">Supplier Name *</label>
                <input type="text" name="supplier_name" id="edit_supplier_name" required>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_contact_person">Contact Person</label>
                    <input type="text" name="contact_person" id="edit_contact_person">
                </div>
                <div class="form-group">
                    <label for="edit_email">Email</label>
                    <input type="email" name="email" id="edit_email">
                </div>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label for="edit_phone">Phone</label>
                    <input type="text" name="phone" id="edit_phone">
                </div>
                <div class="form-group">
                    <label for="edit_address">Address</label>
                    <input type="text" name="address" id="edit_address">
                </div>
            </div>
            <button type="submit" name="edit_supplier" class="btn btn-success">Update Supplier</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<!-- Status Toggle Modal -->
<div id="statusModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Change Supplier Status</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <p>Are you sure you want to <span id="status_action"></span> this supplier?</p>
        <p style="color: #856404; font-weight: 600;">This will affect whether the supplier appears in purchase order forms.</p>
        <form method="post">
            <input type="hidden" name="supplier_id" id="status_supplier_id">
            <input type="hidden" name="current_status" id="current_status">
            <button type="submit" name="toggle_active" class="btn btn-warning">Yes, Change Status</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<script>
function editSupplier(id, name, contact, email, phone, address) {
    document.getElementById('edit_supplier_id').value = id;
    document.getElementById('edit_supplier_name').value = name;
    document.getElementById('edit_contact_person').value = contact;
    document.getElementById('edit_email').value = email;
    document.getElementById('edit_phone').value = phone;
    document.getElementById('edit_address').value = address;
    document.getElementById('editModal').style.display = 'block';
}

function toggleStatus(id, isActive) {
    document.getElementById('status_supplier_id').value = id;
    document.getElementById('current_status').value = isActive ? 'active' : 'inactive';
    document.getElementById('status_action').textContent = isActive ? 'deactivate' : 'activate';
    document.getElementById('statusModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('editModal').style.display = 'none';
    document.getElementById('statusModal').style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    if (event.target.className === 'modal') {
        event.target.style.display = 'none';
    }
}
</script>

<?php include '../includes/admin_footer.php'; ?>
