<?php
require_once 'config.php';

$message = '';
$error = '';

// Get filters
$filter_loan_id = $_GET['loan_id'] ?? '';
$filter_date_from = $_GET['date_from'] ?? '';
$filter_date_to = $_GET['date_to'] ?? '';
$filter_payment_method = $_GET['payment_method'] ?? '';
$search = $_GET['search'] ?? '';

// Build query with remaining balance calculation
$query = "
    SELECT lp.*,
           l.name as loan_name,
           l.original_amount as loan_amount,
           u.username as recorded_by_name,
           (SELECT SUM(lp2.payment_amount)
            FROM loan_payments lp2
            WHERE lp2.loan_id = lp.loan_id
            AND lp2.payment_date <= lp.payment_date
            AND lp2.id <= lp.id) as cumulative_paid,
           (l.original_amount - (SELECT SUM(lp2.payment_amount)
                                FROM loan_payments lp2
                                WHERE lp2.loan_id = lp.loan_id
                                AND lp2.payment_date <= lp.payment_date
                                AND lp2.id <= lp.id)) as remaining_balance
    FROM loan_payments lp
    JOIN liabilities l ON lp.loan_id = l.id
    JOIN users u ON lp.recorded_by = u.id
    WHERE 1=1
";

$params = [];

if ($filter_loan_id) {
    $query .= " AND lp.loan_id = ?";
    $params[] = $filter_loan_id;
}

if ($filter_date_from) {
    $query .= " AND lp.payment_date >= ?";
    $params[] = $filter_date_from;
}

if ($filter_date_to) {
    $query .= " AND lp.payment_date <= ?";
    $params[] = $filter_date_to;
}

if ($filter_payment_method) {
    $query .= " AND lp.payment_method = ?";
    $params[] = $filter_payment_method;
}

if ($search) {
    $query .= " AND (l.name LIKE ? OR lp.payment_amount LIKE ? OR lp.payment_method LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

$query .= " ORDER BY lp.payment_date DESC, lp.created_at DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$payments = $stmt->fetchAll();

// Get loans for filter dropdown
$loans_stmt = $pdo->query("SELECT id, name FROM liabilities ORDER BY name");
$loans = $loans_stmt->fetchAll();

// Calculate totals
$total_payments = count($payments);
$total_amount = array_sum(array_column($payments, 'payment_amount'));

include 'includes/accounting_header.php';
?>

<style>
.payment-history-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.payment-history-header {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.payment-history-header h1 {
    color: #2c3e50;
    margin: 0 0 10px 0;
}

.payment-history-header p {
    color: #7f8c8d;
    margin: 0;
}

.filters-section {
    background: white;
    padding: 20px;
    border-radius: 10px;
    margin-bottom: 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.filters-form {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group input,
.filter-group select {
    padding: 8px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.btn {
    padding: 8px 16px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.2s;
}

.btn-primary {
    background: #007bff;
    color: white;
}

.btn-primary:hover {
    background: #0056b3;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #545b62;
}

.btn-success {
    background: #28a745;
    color: white;
}

.btn-success:hover {
    background: #1e7e34;
}

.summary-section {
    background: white;
    padding: 20px;
    border-radius: 10px;
    margin-bottom: 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    display: flex;
    justify-content: space-around;
    align-items: center;
}

.summary-item {
    text-align: center;
}

.summary-item h3 {
    color: #2c3e50;
    margin: 0 0 5px 0;
    font-size: 1.5em;
}

.summary-item p {
    color: #7f8c8d;
    margin: 0;
}

.payments-table-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.table-container {
    overflow-x: auto;
}

.payments-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
}

.payments-table th,
.payments-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.payments-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
    position: sticky;
    top: 0;
}

.payments-table tr:hover {
    background: #f8f9fa;
}

.amount {
    font-weight: 600;
    color: #2c3e50;
}

.payment-method {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.8em;
    font-weight: bold;
    text-transform: uppercase;
}

.payment-method.cash {
    background: #28a745;
    color: white;
}

.payment-method.bank_transfer {
    background: #007bff;
    color: white;
}

.payment-method.cheque {
    background: #ffc107;
    color: #212529;
}

.payment-method.mobile_money {
    background: #6f42c1;
    color: white;
}

.no-payments {
    text-align: center;
    color: #7f8c8d;
    font-style: italic;
    padding: 40px;
}

.message {
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border-color: #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border-color: #f5c6cb;
}

@media (max-width: 768px) {
    .filters-form {
        grid-template-columns: 1fr;
    }

    .summary-section {
        flex-direction: column;
        gap: 15px;
    }
}
</style>

<div class="payment-history-container">
    <div class="payment-history-header">
        <h1>💳 Loan Payment History</h1>
        <p>Track all loan payment transactions</p>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Filters -->
    <div class="filters-section">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
            <h3 style="margin: 0; color: #2c3e50;">Filter Payments</h3>
            <a href="print_loan_payment_history.php?<?php echo http_build_query($_GET); ?>" class="btn btn-success" target="_blank">
                <i class="fas fa-print"></i> Print Report
            </a>
        </div>
        <form method="get" class="filters-form">
            <div class="filter-group">
                <label for="loan_id">Loan:</label>
                <select name="loan_id" id="loan_id">
                    <option value="">All Loans</option>
                    <?php foreach ($loans as $loan): ?>
                        <option value="<?php echo $loan['id']; ?>" <?php echo $filter_loan_id == $loan['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($loan['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label for="date_from">Date From:</label>
                <input type="date" name="date_from" id="date_from" value="<?php echo htmlspecialchars($filter_date_from); ?>">
            </div>

            <div class="filter-group">
                <label for="date_to">Date To:</label>
                <input type="date" name="date_to" id="date_to" value="<?php echo htmlspecialchars($filter_date_to); ?>">
            </div>

            <div class="filter-group">
                <label for="payment_method">Payment Method:</label>
                <select name="payment_method" id="payment_method">
                    <option value="">All Methods</option>
                    <option value="cash" <?php echo $filter_payment_method == 'cash' ? 'selected' : ''; ?>>Cash</option>
                    <option value="bank_transfer" <?php echo $filter_payment_method == 'bank_transfer' ? 'selected' : ''; ?>>Bank Transfer</option>
                    <option value="cheque" <?php echo $filter_payment_method == 'cheque' ? 'selected' : ''; ?>>Cheque</option>
                    <option value="mobile_money" <?php echo $filter_payment_method == 'mobile_money' ? 'selected' : ''; ?>>Mobile Money</option>
                </select>
            </div>

            <div class="filter-group">
                <label for="search">Search:</label>
                <input type="text" name="search" id="search" placeholder="Loan name, amount, method..." value="<?php echo htmlspecialchars($search); ?>">
            </div>

            <div class="filter-group">
                <button type="submit" class="btn btn-primary">Filter</button>
                <a href="loan_payment_history.php" class="btn btn-secondary">Clear</a>
            </div>
        </form>
    </div>

    <!-- Summary -->
    <div class="summary-section">
        <div class="summary-item">
            <h3><?php echo $total_payments; ?></h3>
            <p>Total Payments</p>
        </div>
        <div class="summary-item">
            <h3>GHC <?php echo number_format($total_amount, 2); ?></h3>
            <p>Total Amount Paid</p>
        </div>
    </div>

    <!-- Payments Table -->
    <div class="payments-table-section">
        <h2>Payment History</h2>

        <div class="table-container">
            <table class="payments-table">
                <thead>
                    <tr>
                        <th>Loan Name</th>
                        <th>Payment Date</th>
                        <th>Amount</th>
                        <th>Payment Method</th>
                        <th>Remaining Balance</th>
                        <th>Recorded By</th>
                        <th>Recorded At</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($payments)): ?>
                        <tr>
                            <td colspan="7" class="no-payments">
                                No payments found matching the selected criteria.
                                <?php if (!$filter_loan_id && !$filter_date_from && !$filter_date_to && !$filter_payment_method && !$search): ?>
                                    <br><br>Start recording payments to see them here.
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($payments as $payment): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($payment['loan_name']); ?></td>
                                <td><?php echo htmlspecialchars($payment['payment_date']); ?></td>
                                <td class="amount">GHC <?php echo number_format($payment['payment_amount'], 2); ?></td>
                                <td>
                                    <span class="payment-method <?php echo str_replace('_', '', $payment['payment_method']); ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $payment['payment_method'])); ?>
                                    </span>
                                </td>
                                <td class="amount">GHC <?php echo number_format($payment['remaining_balance'], 2); ?></td>
                                <td><?php echo htmlspecialchars($payment['recorded_by_name']); ?></td>
                                <td><?php echo htmlspecialchars($payment['created_at']); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<?php
include '../includes/admin_footer.php';
?>
