<?php
require_once '../config.php';

$message = '';
$error = '';

// Get filter parameters
$account_id = $_GET['account_id'] ?? '';
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');
$account_type = $_GET['account_type'] ?? 'all';
$format = $_GET['format'] ?? 'csv';

// Build general ledger query (same as main report)
try {
    $where_conditions = [];
    $params = [];

    // Date range
    $where_conditions[] = "je.transaction_date BETWEEN ? AND ?";
    $params[] = $start_date;
    $params[] = $end_date;

    // Account filter
    if (!empty($account_id)) {
        $where_conditions[] = "jel.account_id = ?";
        $params[] = $account_id;
    }

    // Account type filter
    if ($account_type !== 'all') {
        $where_conditions[] = "coa.account_type = ?";
        $params[] = $account_type;
    }

    $where_clause = implode(' AND ', $where_conditions);

    // Get general ledger entries
    $query = "
        SELECT
            je.id as journal_entry_id,
            je.transaction_date,
            je.description,
            je.reference_number,
            je.source_type,
            jel.account_id,
            jel.debit_amount,
            jel.credit_amount,
            coa.account_code,
            coa.account_name,
            coa.account_type
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE {$where_clause}
        AND je.status = 'posted'
        ORDER BY coa.account_code ASC, je.transaction_date ASC, je.id ASC
    ";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $ledger_entries = $stmt->fetchAll();

    // Group entries by account and calculate running balances
    $grouped_entries = [];
    $account_balances = [];

    foreach ($ledger_entries as $entry) {
        $account_id = $entry['account_id'];

        if (!isset($grouped_entries[$account_id])) {
            $grouped_entries[$account_id] = [
                'account_code' => $entry['account_code'],
                'account_name' => $entry['account_name'],
                'account_type' => $entry['account_type'],
                'entries' => [],
                'opening_balance' => 0,
                'closing_balance' => 0
            ];

            // Calculate opening balance for this account
            $balance_query = "
                SELECT COALESCE(SUM(jel.debit_amount - jel.credit_amount), 0) as opening_balance
                FROM journal_entry_lines jel
                JOIN journal_entries je ON jel.journal_entry_id = je.id
                WHERE jel.account_id = ?
                AND je.transaction_date < ?
                AND je.status = 'posted'
            ";
            $balance_stmt = $pdo->prepare($balance_query);
            $balance_stmt->execute([$account_id, $start_date]);
            $grouped_entries[$account_id]['opening_balance'] = $balance_stmt->fetch()['opening_balance'];
        }

        $grouped_entries[$account_id]['entries'][] = $entry;
    }

    // Calculate running balances for each account
    foreach ($grouped_entries as $account_id => &$account_data) {
        $running_balance = $account_data['opening_balance'];

        foreach ($account_data['entries'] as &$entry) {
            $amount = $entry['debit_amount'] - $entry['credit_amount'];
            $running_balance += $amount;
            $entry['running_balance'] = $running_balance;
        }

        $account_data['closing_balance'] = $running_balance;
    }

} catch (PDOException $e) {
    $error = "Error fetching general ledger data: " . $e->getMessage();
    $grouped_entries = [];
}

// Export based on format
if ($format === 'excel') {
    // Set headers for Excel download
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="general_ledger_' . date('Y-m-d') . '.xls"');
    header('Cache-Control: max-age=0');

    echo "<html><body>";
    echo "<h1>General Ledger Report</h1>";
    echo "<p>Period: " . date('M d, Y', strtotime($start_date)) . " to " . date('M d, Y', strtotime($end_date)) . "</p>";
    echo "<table border='1'>";

    foreach ($grouped_entries as $account_id => $account_data) {
        // Account header
        echo "<tr><td colspan='7'><strong>{$account_data['account_code']} - {$account_data['account_name']} ({$account_data['account_type']})</strong></td></tr>";
        echo "<tr><td colspan='7'>Opening Balance: " . number_format($account_data['opening_balance'], 2) . " | Closing Balance: " . number_format($account_data['closing_balance'], 2) . "</td></tr>";

        // Table headers
        echo "<tr>";
        echo "<th>Date</th>";
        echo "<th>Reference</th>";
        echo "<th>Description</th>";
        echo "<th>Source</th>";
        echo "<th>Debit (GHC)</th>";
        echo "<th>Credit (GHC)</th>";
        echo "<th>Balance (GHC)</th>";
        echo "</tr>";

        // Opening balance row
        echo "<tr>";
        echo "<td>" . date('M d, Y', strtotime($start_date)) . "</td>";
        echo "<td>-</td>";
        echo "<td>Opening Balance</td>";
        echo "<td>-</td>";
        echo "<td>-</td>";
        echo "<td>-</td>";
        echo "<td>" . number_format($account_data['opening_balance'], 2) . "</td>";
        echo "</tr>";

        // Transaction rows
        foreach ($account_data['entries'] as $entry) {
            echo "<tr>";
            echo "<td>" . date('M d, Y', strtotime($entry['transaction_date'])) . "</td>";
            echo "<td>" . htmlspecialchars($entry['reference_number'] ?? $entry['journal_entry_id']) . "</td>";
            echo "<td>" . htmlspecialchars($entry['description']) . "</td>";
            echo "<td>" . htmlspecialchars($entry['source_type']) . "</td>";
            echo "<td>" . ($entry['debit_amount'] > 0 ? number_format($entry['debit_amount'], 2) : '-') . "</td>";
            echo "<td>" . ($entry['credit_amount'] > 0 ? number_format($entry['credit_amount'], 2) : '-') . "</td>";
            echo "<td>" . number_format($entry['running_balance'], 2) . "</td>";
            echo "</tr>";
        }

        // Closing balance row
        echo "<tr>";
        echo "<td>" . date('M d, Y', strtotime($end_date)) . "</td>";
        echo "<td>-</td>";
        echo "<td>Closing Balance</td>";
        echo "<td>-</td>";
        echo "<td>-</td>";
        echo "<td>-</td>";
        echo "<td>" . number_format($account_data['closing_balance'], 2) . "</td>";
        echo "</tr>";

        // Empty row between accounts
        echo "<tr><td colspan='7'></td></tr>";
    }

    echo "</table></body></html>";
    exit;

} elseif ($format === 'pdf') {
    // For PDF, we'll use a simple HTML-based approach that can be printed to PDF
    header('Content-Type: application/pdf');
    header('Content-Disposition: attachment; filename="general_ledger_' . date('Y-m-d') . '.pdf"');

    echo "<html><head><title>General Ledger Report</title></head><body>";
    echo "<h1>General Ledger Report</h1>";
    echo "<p>Period: " . date('M d, Y', strtotime($start_date)) . " to " . date('M d, Y', strtotime($end_date)) . "</p>";
    echo "<style>table { border-collapse: collapse; width: 100%; } th, td { border: 1px solid #ddd; padding: 8px; text-align: left; } th { background-color: #f2f2f2; }</style>";

    foreach ($grouped_entries as $account_id => $account_data) {
        echo "<h2>{$account_data['account_code']} - {$account_data['account_name']}</h2>";
        echo "<p><strong>Account Type:</strong> {$account_data['account_type']}</p>";
        echo "<p><strong>Opening Balance:</strong> " . number_format($account_data['opening_balance'], 2) . " | <strong>Closing Balance:</strong> " . number_format($account_data['closing_balance'], 2) . "</p>";

        echo "<table>";
        echo "<tr><th>Date</th><th>Reference</th><th>Description</th><th>Source</th><th>Debit (GHC)</th><th>Credit (GHC)</th><th>Balance (GHC)</th></tr>";

        // Opening balance
        echo "<tr><td>" . date('M d, Y', strtotime($start_date)) . "</td><td>-</td><td>Opening Balance</td><td>-</td><td>-</td><td>-</td><td>" . number_format($account_data['opening_balance'], 2) . "</td></tr>";

        foreach ($account_data['entries'] as $entry) {
            echo "<tr>";
            echo "<td>" . date('M d, Y', strtotime($entry['transaction_date'])) . "</td>";
            echo "<td>" . htmlspecialchars($entry['reference_number'] ?? $entry['journal_entry_id']) . "</td>";
            echo "<td>" . htmlspecialchars($entry['description']) . "</td>";
            echo "<td>" . htmlspecialchars($entry['source_type']) . "</td>";
            echo "<td>" . ($entry['debit_amount'] > 0 ? number_format($entry['debit_amount'], 2) : '-') . "</td>";
            echo "<td>" . ($entry['credit_amount'] > 0 ? number_format($entry['credit_amount'], 2) : '-') . "</td>";
            echo "<td>" . number_format($entry['running_balance'], 2) . "</td>";
            echo "</tr>";
        }

        // Closing balance
        echo "<tr><td>" . date('M d, Y', strtotime($end_date)) . "</td><td>-</td><td>Closing Balance</td><td>-</td><td>-</td><td>-</td><td>" . number_format($account_data['closing_balance'], 2) . "</td></tr>";
        echo "</table><br>";
    }

    echo "</body></html>";
    exit;

} else {
    // Default CSV export
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="general_ledger_' . date('Y-m-d') . '.csv"');

    $output = fopen('php://output', 'w');

    // CSV headers
    fputcsv($output, ['Account Code', 'Account Name', 'Account Type', 'Date', 'Reference', 'Description', 'Source', 'Debit (GHC)', 'Credit (GHC)', 'Balance (GHC)']);

    foreach ($grouped_entries as $account_id => $account_data) {
        // Account header row
        fputcsv($output, [$account_data['account_code'], $account_data['account_name'], $account_data['account_type'], '', '', '', '', '', '', '']);

        // Opening balance
        fputcsv($output, ['', '', '', date('M d, Y', strtotime($start_date)), '-', 'Opening Balance', '-', '-', '-', number_format($account_data['opening_balance'], 2)]);

        // Transaction rows
        foreach ($account_data['entries'] as $entry) {
            fputcsv($output, [
                '',
                '',
                '',
                date('M d, Y', strtotime($entry['transaction_date'])),
                $entry['reference_number'] ?? $entry['journal_entry_id'],
                $entry['description'],
                $entry['source_type'],
                $entry['debit_amount'] > 0 ? number_format($entry['debit_amount'], 2) : '',
                $entry['credit_amount'] > 0 ? number_format($entry['credit_amount'], 2) : '',
                number_format($entry['running_balance'], 2)
            ]);
        }

        // Closing balance
        fputcsv($output, ['', '', '', date('M d, Y', strtotime($end_date)), '-', 'Closing Balance', '-', '-', '-', number_format($account_data['closing_balance'], 2)]);

        // Empty row
        fputcsv($output, []);
    }

    fclose($output);
    exit;
}
?>
