<?php
require_once '../config.php';

// Get filter values
$filter_class = $_GET['class'] ?? '';
$filter_academic_year = $_GET['academic_year'] ?? '';
$filter_status = $_GET['status'] ?? 'all';
$report_type = $_GET['report_type'] ?? 'summary';

// Get fee statistics
$query = "SELECT
    COUNT(*) as total_fees,
    SUM(CASE WHEN sf.paid = TRUE THEN 1 ELSE 0 END) as paid_fees,
    SUM(CASE WHEN sf.paid_amount > 0 AND sf.paid_amount >= sf.amount THEN 1 ELSE 0 END) as fully_paid_fees,
    SUM(CASE WHEN sf.paid_amount > 0 AND sf.paid_amount < sf.amount THEN 1 ELSE 0 END) as partial_fees,
    SUM(COALESCE(sf.paid_amount, 0)) as total_collected,
    SUM(sf.amount - COALESCE(sf.paid_amount, 0)) as total_outstanding,
    SUM(sf.amount) as total_expected
FROM student_fees sf
JOIN students s ON sf.student_id = s.id
WHERE 1=1";

$params = [];
if ($filter_class) {
    $query .= " AND s.class = ?";
    $params[] = $filter_class;
}
if ($filter_academic_year) {
    $query .= " AND s.academic_year = ?";
    $params[] = $filter_academic_year;
}

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$stats = $stmt->fetch();

// Calculate unpaid fees as total - fully paid - partial
$stats['unpaid_fees'] = $stats['total_fees'] - $stats['fully_paid_fees'] - $stats['partial_fees'];

// Get detailed fee data
$query = "SELECT
    sf.*,
    CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as student_name,
    s.class,
    s.academic_year,
    f.name as fee_name,
    f.amount as fee_amount
FROM student_fees sf
JOIN students s ON sf.student_id = s.id
JOIN fees f ON sf.fee_id = f.id
WHERE 1=1";

$params = [];
if ($filter_class) {
    $query .= " AND s.class = ?";
    $params[] = $filter_class;
}
if ($filter_academic_year) {
    $query .= " AND s.academic_year = ?";
    $params[] = $filter_academic_year;
}

if ($filter_status === 'paid') {
    $query .= " AND sf.paid = TRUE";
} elseif ($filter_status === 'partial') {
    $query .= " AND sf.paid_amount > 0 AND sf.paid_amount < sf.amount AND sf.paid = FALSE";
} elseif ($filter_status === 'unpaid') {
    $query .= " AND sf.paid = FALSE AND (sf.paid_amount IS NULL OR sf.paid_amount = 0)";
}

$query .= " ORDER BY sf.due_date DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$fee_details = $stmt->fetchAll();

// Get payment methods summary
$payment_methods_query = "SELECT
    payment_method,
    COUNT(*) as count,
    SUM(amount) as total_amount
FROM student_fees
WHERE paid = TRUE AND payment_method IS NOT NULL";

if ($filter_class || $filter_academic_year) {
    $payment_methods_query .= " AND student_id IN (
        SELECT id FROM students WHERE 1=1";
    if ($filter_class) $payment_methods_query .= " AND class = '$filter_class'";
    if ($filter_academic_year) $payment_methods_query .= " AND academic_year = '$filter_academic_year'";
    $payment_methods_query .= ")";
}

$payment_methods_query .= " GROUP BY payment_method ORDER BY total_amount DESC";

$payment_methods = $pdo->query($payment_methods_query)->fetchAll();

// Get distinct classes and academic years for filter dropdowns
$classes_stmt = $pdo->query("SELECT name FROM classes ORDER BY name");
$classes = $classes_stmt->fetchAll(PDO::FETCH_COLUMN);

$years_stmt = $pdo->query("SELECT year FROM academic_years ORDER BY year DESC");
$academic_years = $years_stmt->fetchAll(PDO::FETCH_COLUMN);

include 'includes/accounting_header.php';
?>

<style>
/* Summary cards */
.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin: 30px 0;
}

.summary-card {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.summary-card .card-icon {
    font-size: 2.5em;
    margin-bottom: 15px;
}

.summary-card h3 {
    color: #2c3e50;
    font-size: 1.8em;
    margin-bottom: 5px;
}

.summary-card p {
    color: #7f8c8d;
    margin: 0;
}

/* Filter Container Styles */
.filter-container {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    border-radius: 15px;
    padding: 30px;
    margin: 30px 0;
    box-shadow: 0 8px 25px rgba(0,0,0,0.15);
    color: white;
}

.filter-header {
    text-align: center;
    margin-bottom: 25px;
}

.filter-header h3 {
    margin: 0 0 10px;
    font-size: 1.8em;
    font-weight: 700;
    color: white;
}

.filter-header p {
    margin: 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.filter-form {
    background: rgba(255, 255, 255, 0.95);
    border-radius: 12px;
    padding: 25px;
    backdrop-filter: blur(10px);
}

.filter-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 20px;
    margin-bottom: 25px;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-label {
    display: flex;
    align-items: center;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 8px;
    font-size: 0.95em;
}

.filter-label i {
    margin-right: 8px;
    color: #4CAF50;
    font-size: 1.1em;
}

.filter-select {
    padding: 12px 15px;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    font-size: 1em;
    background: white;
    color: #2c3e50;
    transition: all 0.3s ease;
    cursor: pointer;
}

.filter-select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
    transform: translateY(-1px);
}

.filter-select:hover {
    border-color: #4CAF50;
}

.filter-actions {
    display: flex;
    gap: 15px;
    justify-content: center;
    flex-wrap: wrap;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    border: none;
    border-radius: 8px;
    font-size: 1em;
    font-weight: 600;
    text-decoration: none;
    cursor: pointer;
    transition: all 0.3s ease;
    text-align: center;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(76, 175, 80, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(76, 175, 80, 0.4);
}

.btn-outline {
    background: transparent;
    color: #4CAF50;
    border: 2px solid #4CAF50;
}

.btn-outline:hover {
    background: #4CAF50;
    color: white;
    transform: translateY(-2px);
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-2px);
}

/* Tables */
.payments-methods-table, #detailed-table {
    background: white;
    border-radius: 15px;
    overflow: hidden;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
}

.payments-methods-table, #detailed-table {
    width: 100%;
    border-collapse: collapse;
}

.payments-methods-table th, #detailed-table th,
.payments-methods-table td, #detailed-table td {
    padding: 15px 20px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.payments-methods-table th, #detailed-table th {
    background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    font-weight: 600;
    color: #495057;
    font-size: 0.95em;
}

.payments-methods-table tr:hover, #detailed-table tr:hover {
    background: rgba(102, 126, 234, 0.05);
    transition: background-color 0.3s ease;
}

.payment-methods, .detailed-report {
    margin-top: 40px;
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    border-radius: 15px;
    padding: 30px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
    border: 1px solid rgba(255,255,255,0.2);
}

.payment-methods h3, .detailed-report h3 {
    margin: 0 0 25px;
    color: #2c3e50;
    font-size: 1.5em;
    font-weight: 700;
    display: flex;
    align-items: center;
    gap: 10px;
}

.payment-methods h3::before, .detailed-report h3::before {
    content: '📈';
    font-size: 1.3em;
}

.export-buttons {
    margin-bottom: 20px;
    display: flex;
    gap: 15px;
    flex-wrap: wrap;
}

.status {
    padding: 6px 12px;
    border-radius: 20px;
    font-weight: bold;
    font-size: 0.85em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status.paid {
    background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
    color: #155724;
}

.status.unpaid {
    background: linear-gradient(135deg, #f8d7da 0%, #f5c6cb 100%);
    color: #721c24;
}

.status.partial {
    background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
    color: #856404;
}

/* Responsive Design */
@media (max-width: 768px) {
    .filter-container {
        padding: 20px;
        margin: 20px 0;
    }

    .filter-header h3 {
        font-size: 1.5em;
    }

    .filter-form {
        padding: 20px;
    }

    .filter-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }

    .filter-actions {
        flex-direction: column;
        align-items: center;
    }

    .btn {
        width: 100%;
        max-width: 280px;
        justify-content: center;
    }

    .stats-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 20px;
        margin: 20px 0;
    }

    .stat-card {
        padding: 25px 20px;
    }

    .stat-value {
        font-size: 2.2em;
    }
}

@media (max-width: 480px) {
    .stats-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }

    .stat-card {
        padding: 20px;
        min-height: 120px;
    }

    .stat-value {
        font-size: 1.8em;
    }

    .filter-container {
        padding: 15px;
    }

    .filter-header h3 {
        font-size: 1.3em;
    }

    .filter-form {
        padding: 15px;
    }
}

@media print {
    .filter-container, .filter-form, .export-buttons, .btn {
        display: none !important;
    }

    .stats-grid {
        display: none !important;
    }

    .detailed-report {
        padding: 0 !important;
        box-shadow: none !important;
        border: none !important;
    }

    .detailed-report h3 {
        display: block !important;
        text-align: center;
        margin-bottom: 20px;
        font-size: 24px;
    }

    #detailed-table {
        width: 100% !important;
        border-collapse: collapse !important;
        box-shadow: none !important;
        border-radius: 0 !important;
    }

    #detailed-table th, #detailed-table td {
        border: 1px solid #000 !important;
        padding: 8px !important;
        font-size: 12px !important;
    }

    #detailed-table th {
        background: #f0f0f0 !important;
        font-weight: bold !important;
    }

    .status {
        padding: 4px 8px !important;
        border-radius: 10px !important;
        font-size: 10px !important;
        font-weight: bold !important;
    }
}
</style>

<?php
?>

<h2>Fee Payment Reports</h2>

<div class="filter-container">
    <div class="filter-header">
        <h3>📊 Filter Fee Reports</h3>
        <p>Customize your report by selecting specific criteria below</p>
    </div>

    <form method="get" action="fee_reports.php" class="filter-form">
        <div class="filter-grid">
            <div class="filter-group">
                <label for="class" class="filter-label">
                    <i class="fas fa-users"></i> Filter by Class
                </label>
                <select id="class" name="class" class="filter-select">
                    <option value="">All Classes</option>
                    <?php foreach ($classes as $class): ?>
                        <option value="<?php echo htmlspecialchars($class); ?>" <?php if ($class == $filter_class) echo 'selected'; ?>>
                            <?php echo htmlspecialchars($class); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label for="academic_year" class="filter-label">
                    <i class="fas fa-calendar-alt"></i> Academic Year
                </label>
                <select id="academic_year" name="academic_year" class="filter-select">
                    <option value="">All Years</option>
                    <?php foreach ($academic_years as $year): ?>
                        <option value="<?php echo htmlspecialchars($year); ?>" <?php if ($year == $filter_academic_year) echo 'selected'; ?>>
                            <?php echo htmlspecialchars($year); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label for="status" class="filter-label">
                    <i class="fas fa-money-check"></i> Payment Status
                </label>
                <select id="status" name="status" class="filter-select">
                    <option value="all" <?php if ($filter_status == 'all') echo 'selected'; ?>>All Status</option>
                    <option value="paid" <?php if ($filter_status == 'paid') echo 'selected'; ?>>Fully Paid</option>
                    <option value="partial" <?php if ($filter_status == 'partial') echo 'selected'; ?>>Partial Payment</option>
                    <option value="unpaid" <?php if ($filter_status == 'unpaid') echo 'selected'; ?>>Unpaid Only</option>
                </select>
            </div>

            <div class="filter-group">
                <label for="report_type" class="filter-label">
                    <i class="fas fa-chart-bar"></i> Report Type
                </label>
                <select id="report_type" name="report_type" class="filter-select">
                    <option value="summary" <?php if ($report_type == 'summary') echo 'selected'; ?>>Summary View</option>
                    <option value="detailed" <?php if ($report_type == 'detailed') echo 'selected'; ?>>Detailed View</option>
                </select>
            </div>
        </div>

        <div class="filter-actions">
            <button type="submit" class="btn btn-primary">
                <i class="fas fa-search"></i> Generate Report
            </button>
            <a href="fee_reports.php" class="btn btn-outline">
                <i class="fas fa-times"></i> Clear Filters
            </a>
        </div>
    </form>
</div>

<!-- Summary Cards -->
<div class="summary-cards">
    <div class="summary-card">
        <div class="card-icon">📊</div>
        <h3><?php echo number_format($stats['total_fees']); ?></h3>
        <p>Total Fee Records</p>
    </div>

    <div class="summary-card">
        <div class="card-icon">✅</div>
        <h3><?php echo number_format($stats['fully_paid_fees']); ?></h3>
        <p>Fully Paid Fees</p>
    </div>

    <div class="summary-card">
        <div class="card-icon">⏳</div>
        <h3><?php echo number_format($stats['partial_fees']); ?></h3>
        <p>Partial Payments</p>
    </div>

    <div class="summary-card">
        <div class="card-icon">❌</div>
        <h3><?php echo number_format($stats['unpaid_fees']); ?></h3>
        <p>Unpaid Fees</p>
    </div>

    <div class="summary-card">
        <div class="card-icon">💰</div>
        <h3>GHC <?php echo number_format($stats['total_collected'], 2); ?></h3>
        <p>Total Collected</p>
    </div>

    <div class="summary-card">
        <div class="card-icon">📈</div>
        <h3>GHC <?php echo number_format($stats['total_outstanding'], 2); ?></h3>
        <p>Total Outstanding</p>
    </div>

    <div class="summary-card">
        <div class="card-icon">🎯</div>
        <h3>GHC <?php echo number_format($stats['total_expected'], 2); ?></h3>
        <p>Total Expected Fees</p>
    </div>
</div>

<?php if ($report_type === 'summary'): ?>
<!-- Payment Methods Summary -->
<div class="payment-methods">
    <h3>Payment Methods Summary</h3>
    <table class="payments-methods-table">
        <thead>
            <tr>
                <th>Payment Method</th>
                <th>Number of Payments</th>
                <th>Total Amount</th>
                <th>Percentage</th>
            </tr>
        </thead>
        <tbody>
            <?php
            $total_payments = array_sum(array_column($payment_methods, 'count'));
            foreach ($payment_methods as $method):
            ?>
                <tr>
                    <td><?php echo htmlspecialchars($method['payment_method'] ?: 'Not Specified'); ?></td>
                    <td><?php echo number_format($method['count']); ?></td>
                    <td>GHC <?php echo number_format($method['total_amount'], 2); ?></td>
                    <td><?php echo $total_payments > 0 ? round(($method['count'] / $total_payments) * 100, 1) : 0; ?>%</td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>
<?php endif; ?>

<?php if ($report_type === 'detailed'): ?>
<!-- Detailed Fee Report -->
<div class="detailed-report">
    <h3>Detailed Fee Report</h3>
    <div class="export-buttons">
        <button onclick="exportToCSV()" class="btn btn-secondary">Export to CSV</button>
        <button onclick="window.open('print_student_fee_summary.php?class=<?php echo urlencode($filter_class); ?>&academic_year=<?php echo urlencode($filter_academic_year); ?>&status=<?php echo urlencode($filter_status); ?>', '_blank')" class="btn btn-secondary">
            <i class="fas fa-print"></i> Print Report
        </button>
        <button onclick="printDetailedReport()" class="btn btn-secondary">Print Detailed List</button>
    </div>

    <table id="detailed-table">
        <thead>
            <tr>
                <th>Student Name</th>
                <th>Class</th>
                <th>Academic Year</th>
                <th>Fee Name</th>
                <th>Amount</th>
                <th>Due Date</th>
                <th>Payment Date</th>
                <th>Payment Method</th>
                <th>Receipt Number</th>
                <th>Status</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($fee_details as $fee): ?>
                <tr>
                    <td><?php echo htmlspecialchars($fee['student_name']); ?></td>
                    <td><?php echo htmlspecialchars($fee['class']); ?></td>
                    <td><?php echo htmlspecialchars($fee['academic_year']); ?></td>
                    <td><?php echo htmlspecialchars($fee['fee_name']); ?></td>
                    <td>GHC <?php echo number_format($fee['fee_amount'], 2); ?></td>
                    <td><?php echo htmlspecialchars($fee['due_date']); ?></td>
                    <td><?php echo $fee['paid_at'] ? htmlspecialchars($fee['paid_at']) : '-'; ?></td>
                    <td><?php echo htmlspecialchars($fee['payment_method'] ?: '-'); ?></td>
                    <td><?php echo htmlspecialchars($fee['receipt_number'] ?: '-'); ?></td>
                    <td>
                        <?php
                        $status_class = 'unpaid';
                        $status_text = 'Unpaid';

                        if ($fee['paid']) {
                            $status_class = 'paid';
                            $status_text = 'Paid';
                        } elseif ($fee['paid_amount'] > 0 && $fee['paid_amount'] < $fee['amount']) {
                            $status_class = 'partial';
                            $status_text = 'Partial';
                        }
                        ?>
                        <span class="status <?php echo $status_class; ?>">
                            <?php echo $status_text; ?>
                        </span>
                    </td>
                    <td>
                        <?php if ($fee['paid'] || $fee['paid_amount'] > 0): ?>
                            <a href="print_receipt.php?fee=<?php echo $fee['id']; ?>" target="_blank" class="btn btn-secondary">
                                <i class="fas fa-print"></i> Print Receipt
                            </a>
                        <?php else: ?>
                            -
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>
<?php endif; ?>

<script>
function exportToCSV() {
    const table = document.getElementById('detailed-table');
    let csv = [];

    // Get headers
    const headers = [];
    for (let i = 0; i < table.rows[0].cells.length; i++) {
        headers.push(table.rows[0].cells[i].textContent);
    }
    csv.push(headers.join(','));

    // Get data
    for (let i = 1; i < table.rows.length; i++) {
        const row = [];
        for (let j = 0; j < table.rows[i].cells.length; j++) {
            row.push('"' + table.rows[i].cells[j].textContent.replace(/"/g, '""') + '"');
        }
        csv.push(row.join(','));
    }

    // Download CSV
    const csvContent = csv.join('\n');
    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'fee_report_<?php echo date('Y-m-d'); ?>.csv';
    a.click();
    window.URL.revokeObjectURL(url);
}

function printDetailedReport() {
    // Hide elements not needed for detailed print
    const elementsToHide = document.querySelector('.filter-container');
    const statsGrid = document.querySelector('.stats-grid');
    const exportButtons = document.querySelector('.export-buttons');
    const h3Title = document.querySelector('.detailed-report h3');
    
    const originalDisplay = {
        filter: elementsToHide.style.display,
        stats: statsGrid.style.display,
        buttons: exportButtons.style.display,
        title: h3Title.style.display
    };
    
    // Hide unwanted elements
    elementsToHide.style.display = 'none';
    statsGrid.style.display = 'none';
    exportButtons.style.display = 'none';
    h3Title.style.display = 'block'; // Ensure title shows
    
    // Add temporary title if needed
    const printTitle = document.createElement('h2');
    printTitle.textContent = 'Detailed Fee Report';
    printTitle.style.textAlign = 'center';
    printTitle.style.marginBottom = '20px';
    document.querySelector('.detailed-report').insertBefore(printTitle, document.querySelector('#detailed-table'));
    
    window.print();
    
    // Restore elements
    elementsToHide.style.display = originalDisplay.filter;
    statsGrid.style.display = originalDisplay.stats;
    exportButtons.style.display = originalDisplay.buttons;
    h3Title.style.display = originalDisplay.title;
    
    // Remove temporary title
    printTitle.remove();
}
</script>


<?php
include 'includes/accounting_footer.php';
?>
