<?php
require_once dirname(__DIR__) . '/config.php';

// Get filter parameters
$year = $_GET['year'] ?? date('Y');

// Calculate year boundaries
$start_date = $year . '-01-01';
$end_date = $year . '-12-31';
$prev_year = $year - 1;
$prev_start_date = $prev_year . '-01-01';
$prev_end_date = $prev_year . '-12-31';

// Initialize variables
$current_year_data = [];
$previous_year_data = [];
$changes = [];

try {
    // Get current year balances (end of year)
    $stmt = $pdo->prepare("
        SELECT
            coa.account_type,
            coa.account_code,
            coa.account_name,
            COALESCE(
                SUM(
                    CASE
                        WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                        ELSE jel.credit_amount - jel.debit_amount
                    END
                ), 0
            ) as balance
        FROM chart_of_accounts coa
        LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
        LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
        WHERE coa.is_active = TRUE
        GROUP BY coa.account_type, coa.account_code, coa.account_name
        ORDER BY coa.account_code
    ");
    $stmt->execute([$end_date]);
    $current_year_raw = $stmt->fetchAll();

    // Get previous year balances (end of previous year)
    $stmt->execute([$prev_end_date]);
    $previous_year_raw = $stmt->fetchAll();

    // Aggregate by account type
    $current_year_data = [
        'assets' => 0,
        'liabilities' => 0,
        'equity' => 0,
        'revenue' => 0,
        'expense' => 0
    ];

    $previous_year_data = [
        'assets' => 0,
        'liabilities' => 0,
        'equity' => 0,
        'revenue' => 0,
        'expense' => 0
    ];

    foreach ($current_year_raw as $account) {
        $type = $account['account_type'];
        if (isset($current_year_data[$type])) {
            $current_year_data[$type] += $account['balance'];
        }
    }

    foreach ($previous_year_raw as $account) {
        $type = $account['account_type'];
        if (isset($previous_year_data[$type])) {
            $previous_year_data[$type] += $account['balance'];
        }
    }

    // Calculate net income for current year
    $current_net_income = $current_year_data['revenue'] - $current_year_data['expense'];

    // Calculate net income for previous year
    $previous_net_income = $previous_year_data['revenue'] - $previous_year_data['expense'];

    // Adjust equity for net income
    $current_year_data['equity'] += $current_net_income;
    $previous_year_data['equity'] += $previous_net_income;

    // Calculate changes
    $changes = [
        'assets' => $current_year_data['assets'] - $previous_year_data['assets'],
        'liabilities' => $current_year_data['liabilities'] - $previous_year_data['liabilities'],
        'equity' => $current_year_data['equity'] - $previous_year_data['equity'],
        'net_income' => $current_net_income - $previous_net_income
    ];

    // Get beginning balances (start of current year)
    $stmt = $pdo->prepare("
        SELECT
            coa.account_type,
            COALESCE(
                SUM(
                    CASE
                        WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                        ELSE jel.credit_amount - jel.debit_amount
                    END
                ), 0
            ) as balance
        FROM chart_of_accounts coa
        LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
        LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date < ?
        WHERE coa.is_active = TRUE
        GROUP BY coa.account_type
    ");
    $stmt->execute([$start_date]);
    $beginning_raw = $stmt->fetchAll();

    $beginning_balances = [
        'assets' => 0,
        'liabilities' => 0,
        'equity' => 0
    ];

    foreach ($beginning_raw as $account) {
        $type = $account['account_type'];
        if (isset($beginning_balances[$type])) {
            $beginning_balances[$type] += $account['balance'];
        }
    }

    // Calculate beginning equity with previous year's net income
    $beginning_balances['equity'] += $previous_net_income;

} catch (PDOException $e) {
    // Handle error - set headers for CSV and output error
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="year_end_summary_error_' . date('Y-m-d') . '.csv"');
    $output = fopen('php://output', 'w');
    fputcsv($output, ['Error', 'Failed to fetch year-end summary data: ' . $e->getMessage()]);
    fclose($output);
    exit();
}

// Set headers for CSV download
$filename = "year_end_summary_" . $year . "_vs_" . $prev_year . "_" . date('Y-m-d') . ".csv";
header('Content-Type: text/csv');
header('Content-Disposition: attachment; filename="' . $filename . '"');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Open output stream
$output = fopen('php://output', 'w');

// Write CSV headers and data
fputcsv($output, ['Year-End Summary Report']);
fputcsv($output, ['Year', $year . ' vs ' . $prev_year]);
fputcsv($output, ['Generated on', date('Y-m-d H:i:s')]);
fputcsv($output, []);

// Year-End Summary Section
fputcsv($output, ['YEAR-END SUMMARY']);
fputcsv($output, ['Category', 'Beginning ' . $year, 'Changes ' . $year, 'Ending ' . $year]);

fputcsv($output, [
    'Assets',
    number_format($beginning_balances['assets'], 2),
    number_format($changes['assets'], 2),
    number_format($current_year_data['assets'], 2)
]);

fputcsv($output, [
    'Liabilities',
    number_format($beginning_balances['liabilities'], 2),
    number_format($changes['liabilities'], 2),
    number_format($current_year_data['liabilities'], 2)
]);

fputcsv($output, [
    'Equity',
    number_format($beginning_balances['equity'], 2),
    number_format($changes['net_income'], 2),
    number_format($current_year_data['equity'], 2)
]);

fputcsv($output, [
    'Total',
    number_format($beginning_balances['assets'] - $beginning_balances['liabilities'] - $beginning_balances['equity'], 2),
    number_format($changes['assets'] - $changes['liabilities'] - $changes['net_income'], 2),
    number_format($current_year_data['assets'] - $current_year_data['liabilities'] - $current_year_data['equity'], 2)
]);

fputcsv($output, []);

// Year-over-Year Comparison Section
fputcsv($output, ['YEAR-OVER-YEAR COMPARISON']);
fputcsv($output, ['Category', $prev_year . ' Ending', $year . ' Ending', 'Change', '% Change']);

fputcsv($output, [
    'Assets',
    number_format($previous_year_data['assets'], 2),
    number_format($current_year_data['assets'], 2),
    number_format($changes['assets'], 2),
    $previous_year_data['assets'] != 0 ? number_format(($changes['assets'] / $previous_year_data['assets']) * 100, 1) . '%' : 'N/A'
]);

fputcsv($output, [
    'Liabilities',
    number_format($previous_year_data['liabilities'], 2),
    number_format($current_year_data['liabilities'], 2),
    number_format($changes['liabilities'], 2),
    $previous_year_data['liabilities'] != 0 ? number_format(($changes['liabilities'] / $previous_year_data['liabilities']) * 100, 1) . '%' : 'N/A'
]);

fputcsv($output, [
    'Equity',
    number_format($previous_year_data['equity'], 2),
    number_format($current_year_data['equity'], 2),
    number_format($changes['equity'], 2),
    $previous_year_data['equity'] != 0 ? number_format(($changes['equity'] / $previous_year_data['equity']) * 100, 1) . '%' : 'N/A'
]);

fputcsv($output, [
    'Net Assets',
    number_format($previous_year_data['assets'] - $previous_year_data['liabilities'] - $previous_year_data['equity'], 2),
    number_format($current_year_data['assets'] - $current_year_data['liabilities'] - $current_year_data['equity'], 2),
    number_format($changes['assets'] - $changes['liabilities'] - $changes['equity'], 2),
    'N/A'
]);

fputcsv($output, [
    'Net Income',
    number_format($previous_net_income, 2),
    number_format($current_net_income, 2),
    number_format($changes['net_income'], 2),
    $previous_net_income != 0 ? number_format(($changes['net_income'] / $previous_net_income) * 100, 1) . '%' : 'N/A'
]);

fputcsv($output, []);

// Financial Health Analysis Section
fputcsv($output, ['FINANCIAL HEALTH ANALYSIS']);
fputcsv($output, ['Metric', 'Value', 'Analysis']);

$growth_rate = $previous_year_data['assets'] != 0 ? number_format(($changes['assets'] / $previous_year_data['assets']) * 100, 1) . '%' : 'N/A';
fputcsv($output, [
    'Asset Growth Rate',
    $growth_rate,
    'Asset growth from ' . $prev_year . ' to ' . $year
]);

$roa = $current_year_data['assets'] != 0 ? number_format(($current_net_income / $current_year_data['assets']) * 100, 1) . '%' : 'N/A';
fputcsv($output, [
    'Return on Assets (ROA)',
    $roa,
    'Profitability for ' . $year
]);

$leverage = $current_year_data['assets'] > 0 ? number_format(($current_year_data['liabilities'] / $current_year_data['assets']) * 100, 1) . '%' : 'N/A';
fputcsv($output, [
    'Debt-to-Asset Ratio',
    $leverage,
    'Leverage ratio for ' . $year
]);

$equity_growth = $previous_year_data['equity'] != 0 ? number_format(($changes['equity'] / $previous_year_data['equity']) * 100, 1) . '%' : 'N/A';
fputcsv($output, [
    'Equity Growth',
    $equity_growth,
    'Equity growth from ' . $prev_year . ' to ' . $year
]);

fclose($output);
?>
