<?php
require_once '../config.php';
require_once '../includes/access_control.php';

// Access control temporarily disabled for expense list page
// requireFeatureAccess('accounts_management');

include 'includes/accounting_header.php';

$message = '';
$error = '';

// Handle delete action
if (isset($_GET['delete']) && is_numeric($_GET['delete'])) {
    try {
        $stmt = $pdo->prepare("DELETE FROM expenses WHERE id = ?");
        $stmt->execute([$_GET['delete']]);
        $message = "Expense deleted successfully!";
    } catch (PDOException $e) {
        $error = "Error deleting expense: " . $e->getMessage();
    }
}

// Get filter parameters
$search = $_GET['search'] ?? '';
$category_filter = $_GET['category'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$sort_by = $_GET['sort_by'] ?? 'expense_date';
$sort_order = $_GET['sort_order'] ?? 'DESC';

// Build query
$where_conditions = [];
$params = [];

if (!empty($search)) {
    $where_conditions[] = "(e.title LIKE ? OR e.vendor_name LIKE ? OR e.description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($category_filter)) {
    $where_conditions[] = "e.category_id = ?";
    $params[] = $category_filter;
}

if (!empty($date_from)) {
    $where_conditions[] = "e.expense_date >= ?";
    $params[] = $date_from;
}

if (!empty($date_to)) {
    $where_conditions[] = "e.expense_date <= ?";
    $params[] = $date_to;
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get total count for pagination
try {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total
        FROM expenses e
        LEFT JOIN budget_categories bc ON e.category_id = bc.id
        $where_clause
    ");
    $stmt->execute($params);
    $total_expenses = $stmt->fetch()['total'];
    $total_pages = ceil($total_expenses / 20);
    $current_page = max(1, min($total_pages, $_GET['page'] ?? 1));
    $offset = ($current_page - 1) * 20;

    // Get expenses with pagination
    $stmt = $pdo->prepare("
        SELECT e.*, bc.name as category_name, bc.is_system, u.username as created_by_name
        FROM expenses e
        LEFT JOIN budget_categories bc ON e.category_id = bc.id
        LEFT JOIN users u ON e.created_by = u.id
        $where_clause
        ORDER BY e.$sort_by $sort_order
        LIMIT ? OFFSET ?
    ");
    $i = 1;
    foreach ($params as $param) {
        $stmt->bindValue($i++, $param);
    }
    $stmt->bindValue($i++, 20, PDO::PARAM_INT);
    $stmt->bindValue($i++, $offset, PDO::PARAM_INT);
    $stmt->execute();
    $expenses = $stmt->fetchAll();

    // Get category totals
    $stmt = $pdo->prepare("
        SELECT bc.name as category_name, SUM(e.amount) as total_amount, COUNT(*) as expense_count
        FROM expenses e
        LEFT JOIN budget_categories bc ON e.category_id = bc.id
        $where_clause
        GROUP BY e.category_id, bc.name
        ORDER BY total_amount DESC
    ");
    $stmt->execute($params);
    $category_totals = $stmt->fetchAll();

    // Get total expenses amount
    $stmt = $pdo->prepare("
        SELECT SUM(e.amount) as total_amount
        FROM expenses e
        LEFT JOIN budget_categories bc ON e.category_id = bc.id
        $where_clause
    ");
    $stmt->execute($params);
    $total_amount = $stmt->fetch()['total_amount'] ?? 0;

} catch (PDOException $e) {
    $error = "Error loading expenses: " . $e->getMessage();
    $expenses = [];
    $category_totals = [];
    $total_amount = 0;
}

// Get categories for filter (both system and user-defined, excluding revenue)
try {
    $stmt = $pdo->query("
        SELECT id, name, is_system,
               CASE WHEN is_system = 1 THEN 'System' ELSE 'Custom' END as type
        FROM budget_categories
        WHERE name != 'Revenue'
        ORDER BY is_system DESC, name ASC
    ");
    $categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $categories = [];
}
?>

<style>
        .expense-list-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 0;
            background: transparent;
            min-height: auto;
            box-shadow: none;
        }

        .expense-header {
            background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
            color: white;
            padding: 30px;
            border-radius: 12px;
            margin-bottom: 30px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .expense-header h1 {
            margin: 0;
            font-size: 2.2em;
            font-weight: 300;
        }

        .expense-header p {
            margin: 5px 0 0 0;
            opacity: 0.9;
            font-size: 1.1em;
        }

        .expense-content {
            display: grid;
            grid-template-columns: 1fr 320px;
            gap: 30px;
        }

        .expense-main {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 15px rgba(0,0,0,0.08);
            border: 1px solid #e1e5e9;
        }

        .table-container {
            overflow-x: auto;
            -webkit-overflow-scrolling: touch;
        }

        .expense-sidebar {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 15px rgba(0,0,0,0.08);
            padding: 25px;
            border: 1px solid #e1e5e9;
        }

        .filters-section {
            background: #f8f9fa;
            padding: 25px;
            border-bottom: 1px solid #dee2e6;
        }

        .filters-form {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
            gap: 20px;
            align-items: end;
        }

        .filter-group {
            display: flex;
            flex-direction: column;
        }

        .filter-group label {
            font-weight: 600;
            margin-bottom: 8px;
            color: #495057;
            font-size: 0.95em;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .filter-group input,
        .filter-group select {
            padding: 12px 16px;
            border: 2px solid #2E7D32;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s ease;
        }

        .filter-group input:focus,
        .filter-group select:focus {
            outline: none;
            border-color: #4CAF50;
            box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
        }

        .filter-actions {
            display: flex;
            gap: 12px;
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            text-decoration: none;
            display: inline-block;
            transition: all 0.3s ease;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .btn-primary {
            background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(46, 125, 50, 0.3);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(46, 125, 50, 0.4);
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #5a6268;
            transform: translateY(-1px);
        }

        .btn-success {
            background: linear-gradient(135deg, #F9A825 0%, #FFC107 100%);
            color: #2E7D32;
            box-shadow: 0 4px 15px rgba(249, 168, 37, 0.3);
        }

        .btn-success:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(249, 168, 37, 0.4);
        }

        .btn-danger {
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            color: white;
        }

        .btn-danger:hover {
            background: linear-gradient(135deg, #c82333 0%, #e8680d 100%);
            transform: translateY(-1px);
        }

        .expenses-table {
            width: 100%;
            border-collapse: collapse;
        }

        .expenses-table th,
        .expenses-table td {
            padding: 16px 20px;
            text-align: left;
            border-bottom: 1px solid #e9ecef;
        }

        .expenses-table th {
            background: #fff;
            font-weight: 700;
            color: #000;
            position: sticky;
            top: 0;
            text-transform: uppercase;
            font-size: 0.85em;
            letter-spacing: 0.5px;
        }

        .expenses-table tr:hover {
            background: #f8f9fa;
        }

        .expense-title {
            font-weight: 600;
            color: #2c3e50;
            max-width: 200px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }

        .expense-amount {
            font-weight: 700;
            color: #e74c3c;
            font-size: 1.1em;
        }

        .expense-category {
            background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
            color: white;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.8em;
            display: inline-block;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .expense-date {
            color: #6c757d;
            font-size: 0.9em;
            font-weight: 500;
        }

        .expense-vendor {
            color: #495057;
            font-size: 0.9em;
            max-width: 150px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
            font-weight: 500;
        }

        .actions-cell {
            white-space: nowrap;
        }

        .action-btn {
            padding: 6px 12px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-size: 0.85em;
            margin-right: 8px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            transition: all 0.3s ease;
        }

        .edit-btn {
            background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%);
            color: #212529;
        }

        .edit-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 10px rgba(255, 193, 7, 0.3);
        }

        .delete-btn {
            background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
            color: white;
        }

        .delete-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 10px rgba(220, 53, 69, 0.3);
        }

        .summary-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
            gap: 20px;
            margin-bottom: 25px;
        }

        .summary-card {
            background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
            padding: 20px;
            border-radius: 12px;
            text-align: center;
            border: 1px solid #e1e5e9;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            transition: transform 0.3s ease;
        }

        .summary-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
        }

        .summary-card h3 {
            margin: 0 0 8px 0;
            font-size: 2em;
            color: #2c3e50;
            font-weight: 700;
        }

        .summary-card p {
            margin: 0;
            color: #6c757d;
            font-size: 0.95em;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .pagination {
            display: flex;
            justify-content: center;
            gap: 8px;
            margin-top: 25px;
            padding-top: 25px;
            border-top: 1px solid #dee2e6;
        }

        .pagination a,
        .pagination span {
            padding: 10px 16px;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            text-decoration: none;
            color: #2E7D32;
            font-weight: 600;
            transition: all 0.3s ease;
        }

        .pagination .active {
            background: #2E7D32;
            color: white;
            border-color: #2E7D32;
        }

        .pagination .disabled {
            color: #6c757d;
            cursor: not-allowed;
        }

        .message {
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 8px;
            font-weight: 600;
        }

        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .no-expenses {
            text-align: center;
            padding: 50px 20px;
            color: #6c757d;
        }

        .no-expenses h3 {
            margin: 0 0 10px 0;
            color: #495057;
        }

        .status-badge {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 0.8em;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: inline-block;
        }

        .status-pending {
            background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%);
            color: #212529;
        }

        .status-approved {
            background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
            color: white;
        }

        .status-rejected {
            background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
            color: white;
        }

        .status-paid {
            background: linear-gradient(135deg, #007bff 0%, #6610f2 100%);
            color: white;
        }

        .status-cancelled {
            background: linear-gradient(135deg, #6c757d 0%, #343a40 100%);
            color: white;
        }
    </style>

    <div class="expense-list-container">
        <div class="expense-header">
            <div>
                <h1>📋 Expense Management</h1>
                <p>View, filter, and manage all school expenses</p>
            </div>
            <div>
                <a href="record_expense.php" class="btn btn-success">➕ Add New Expense</a>
                <a href="index.php" class="btn btn-secondary">🏠 Dashboard</a>
            </div>
        </div>

        <?php if ($message): ?>
            <div class="message success"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="message error"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <div class="expense-content">
            <div class="expense-main">
                <!-- Summary Cards -->
                <div class="summary-cards">
                    <div class="summary-card">
                        <h3>GHC <?php echo number_format($total_amount, 2); ?></h3>
                        <p>Total Expenses</p>
                    </div>
                    <div class="summary-card">
                        <h3><?php echo count($expenses); ?></h3>
                        <p>Current Page</p>
                    </div>
                    <div class="summary-card">
                        <h3><?php echo $total_expenses; ?></h3>
                        <p>Total Records</p>
                    </div>
                    <div class="summary-card">
                        <h3><?php echo $total_pages; ?></h3>
                        <p>Total Pages</p>
                    </div>
                </div>

                <!-- Filters -->
                <div class="filters-section">
                    <form method="get" class="filters-form">
                        <div class="filter-group">
                            <label for="search">Search</label>
                            <input type="text" name="search" id="search" value="<?php echo htmlspecialchars($search); ?>"
                                   placeholder="Title, vendor, or description...">
                        </div>

                        <div class="filter-group">
                            <label for="category">Category</label>
                            <select name="category" id="category">
                                <option value="">All Categories</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>"
                                            <?php echo $category_filter == $category['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($category['name']); ?>
                                        <?php if (!$category['is_system']): ?>
                                            <span style="color: #28a745; font-size: 0.8em;">(Custom)</span>
                                        <?php endif; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="filter-group">
                            <label for="date_from">From Date</label>
                            <input type="date" name="date_from" id="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                        </div>

                        <div class="filter-group">
                            <label for="date_to">To Date</label>
                            <input type="date" name="date_to" id="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                        </div>

                        <div class="filter-actions">
                            <button type="submit" class="btn btn-primary">🔍 Filter</button>
                            <a href="expense_list.php" class="btn btn-secondary">🔄 Reset</a>
                        </div>
                    </form>
                </div>

                <!-- Expenses Table -->
                <?php if (empty($expenses)): ?>
                    <div class="no-expenses">
                        <h3>No expenses found</h3>
                        <p>Try adjusting your filters or <a href="record_expense.php">record a new expense</a>.</p>
                    </div>
                <?php else: ?>
                    <div class="table-container">
                        <table class="expenses-table">
                        <thead>
                            <tr>
                        <th>Title</th>
                        <th>Amount</th>
                        <th>Category</th>
                        <th>Date</th>
                        <th>Vendor</th>
                        <th>Payment Method</th>
                        <th>Status</th>
                        <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($expenses as $expense): ?>
                                <tr>
                                    <td>
                                        <div class="expense-title" title="<?php echo htmlspecialchars($expense['title']); ?>">
                                            <?php echo htmlspecialchars($expense['title']); ?>
                                        </div>
                                        <?php if (!empty($expense['description'])): ?>
                                            <div style="font-size: 0.8em; color: #6c757d; margin-top: 2px;">
                                                <?php echo htmlspecialchars(substr($expense['description'], 0, 50)) . (strlen($expense['description']) > 50 ? '...' : ''); ?>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td class="expense-amount">
                                        GHC <?php echo number_format($expense['amount'], 2); ?>
                                    </td>
                                    <td>
                                        <span class="expense-category">
                                            <?php echo htmlspecialchars($expense['category_name'] ?? 'Uncategorized'); ?>
                                            <?php if (isset($expense['is_system']) && !$expense['is_system']): ?>
                                                <span style="color: #28a745; font-size: 0.8em;">(Custom)</span>
                                            <?php endif; ?>
                                        </span>
                                    </td>
                                    <td class="expense-date">
                                        <?php echo date('M d, Y', strtotime($expense['expense_date'])); ?>
                                    </td>
                                    <td>
                                        <div class="expense-vendor" title="<?php echo htmlspecialchars($expense['vendor_name'] ?? ''); ?>">
                                            <?php echo htmlspecialchars($expense['vendor_name'] ?? 'N/A'); ?>
                                        </div>
                                    </td>
                                    <td>
                                        <span style="background: #f8f9fa; padding: 2px 8px; border-radius: 12px; font-size: 0.8em;">
                                            <?php echo ucfirst(str_replace('_', ' ', $expense['payment_method'] ?? 'unknown')); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="status-badge status-<?php echo $expense['status'] ?? 'pending'; ?>">
                                            <?php echo ucfirst($expense['status'] ?? 'pending'); ?>
                                        </span>
                                    </td>
                                    <td class="actions-cell">
                                        <button class="action-btn edit-btn" onclick="editExpense(<?php echo $expense['id']; ?>)">
                                            ✏️ Edit
                                        </button>
                                        <a href="?delete=<?php echo $expense['id']; ?>&<?php echo http_build_query(array_diff_key($_GET, ['delete' => ''])); ?>"
                                           class="action-btn delete-btn"
                                           onclick="return confirm('Are you sure you want to delete this expense?')">
                                            🗑️ Delete
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    </div>

                    <!-- Pagination -->
                    <?php if ($total_pages > 1): ?>
                        <div class="pagination">
                            <?php if ($current_page > 1): ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $current_page - 1])); ?>">Previous</a>
                            <?php else: ?>
                                <span class="disabled">Previous</span>
                            <?php endif; ?>

                            <?php for ($i = max(1, $current_page - 2); $i <= min($total_pages, $current_page + 2); $i++): ?>
                                <?php if ($i == $current_page): ?>
                                    <span class="active"><?php echo $i; ?></span>
                                <?php else: ?>
                                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                                <?php endif; ?>
                            <?php endfor; ?>

                            <?php if ($current_page < $total_pages): ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $current_page + 1])); ?>">Next</a>
                            <?php else: ?>
                                <span class="disabled">Next</span>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>

            <div class="expense-sidebar">
                <h3 style="margin-top: 0; color: #2c3e50;">📊 Category Summary</h3>
                <?php if (empty($category_totals)): ?>
                    <p style="color: #7f8c8d; font-style: italic;">No expenses to summarize</p>
                <?php else: ?>
                    <div style="max-height: 300px; overflow-y: auto;">
                        <?php foreach ($category_totals as $category): ?>
                            <div style="display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #f0f0f0;">
                                <div>
                                    <strong><?php echo htmlspecialchars($category['category_name']); ?></strong>
                                    <div style="font-size: 0.8em; color: #7f8c8d;">
                                        <?php echo $category['expense_count']; ?> expenses
                                    </div>
                                </div>
                                <div style="text-align: right;">
                                    <div style="font-weight: 600; color: #e74c3c;">
                                        GHC <?php echo number_format($category['total_amount'], 2); ?>
                                    </div>
                                    <div style="font-size: 0.8em; color: #7f8c8d;">
                                        <?php echo $total_amount > 0 ? round(($category['total_amount'] / $total_amount) * 100, 1) : 0; ?>%
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>

                <h3 style="margin-top: 30px; color: #2c3e50;">🔗 Quick Actions</h3>
                <div style="display: flex; flex-direction: column; gap: 10px;">
                    <a href="record_expense.php" class="btn btn-success" style="text-align: center;">Add New Expense</a>
                    <a href="balance_sheet_report.php" class="btn btn-primary" style="text-align: center;">View Balance Sheet</a>
                    <a href="profit_loss_report.php" class="btn btn-primary" style="text-align: center;">View P&L Report</a>
                    <a href="cash_flow_statement.php" class="btn btn-primary" style="text-align: center;">View Cash Flow</a>
                </div>
            </div>
        </div>
    </div>

    <script>
        function editExpense(id) {
            // For now, just redirect to record expense with the ID
            // In a full implementation, you'd load the expense data into a modal or separate edit form
            window.location.href = 'record_expense.php?edit=' + id;
        }
    </script>

<?php include 'includes/accounting_footer.php'; ?>
</body>
</html>
