<?php
require_once '../config.php';

$message = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'save_report') {
        try {
            $stmt = $pdo->prepare("
                INSERT INTO custom_reports (
                    name, description, report_type, query_config, filters, columns_config, created_by, is_public
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $_POST['report_name'],
                $_POST['report_description'],
                $_POST['report_type'],
                json_encode($_POST['query_config'] ?? []),
                json_encode($_POST['filters'] ?? []),
                json_encode($_POST['columns_config'] ?? []),
                $_SESSION['user_id'],
                isset($_POST['is_public']) ? 1 : 0
            ]);

            $message = "Custom report saved successfully!";
        } catch (PDOException $e) {
            $error = "Error saving report: " . $e->getMessage();
        }
    } elseif ($action === 'run_report') {
        // Handle report execution
        $report_id = $_POST['report_id'] ?? 0;
        if ($report_id) {
            try {
                $stmt = $pdo->prepare("SELECT * FROM custom_reports WHERE id = ?");
                $stmt->execute([$report_id]);
                $report = $stmt->fetch();

                if ($report) {
                    $query_config = json_decode($report['query_config'], true);
                    $filters = json_decode($report['filters'], true);
                    $columns_config = json_decode($report['columns_config'], true);

                    // Build dynamic query based on configuration
                    $query = buildCustomQuery($query_config, $filters, $columns_config);
                    $stmt = $pdo->prepare($query);
                    $stmt->execute();
                    $report_data = $stmt->fetchAll();
                }
            } catch (PDOException $e) {
                $error = "Error running report: " . $e->getMessage();
            }
        }
    }
}

// Get saved reports
try {
    $stmt = $pdo->prepare("
        SELECT cr.*, u.username as created_by_name
        FROM custom_reports cr
        LEFT JOIN users u ON cr.created_by = u.id
        WHERE cr.created_by = ? OR cr.is_public = 1
        ORDER BY cr.created_at DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $saved_reports = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error fetching saved reports: " . $e->getMessage();
}

// Get available tables and columns for report builder
$available_tables = [
    'student_fees' => ['id', 'student_id', 'fee_id', 'amount', 'paid_amount', 'balance', 'paid_at', 'created_at'],
    'expenses' => ['id', 'title', 'amount', 'category_id', 'expense_date', 'payment_method', 'vendor_name', 'description', 'created_at'],
    'budgets' => ['id', 'name', 'academic_year_id', 'category_id', 'budget_type', 'planned_amount', 'description', 'created_at'],
    'budget_categories' => ['id', 'name', 'description', 'parent_id'],
    'assets' => ['id', 'name', 'asset_type', 'purchase_date', 'purchase_cost', 'current_value', 'depreciation_rate', 'status', 'created_at'],
    'liabilities' => ['id', 'name', 'liability_type', 'original_amount', 'current_balance', 'interest_rate', 'start_date', 'end_date', 'status', 'created_at'],
    'incomes' => ['id', 'title', 'amount', 'income_type', 'income_date', 'source', 'description', 'created_at'],
    'payroll' => ['id', 'staff_id', 'basic_salary', 'allowances', 'deductions', 'net_salary', 'payroll_date', 'status', 'created_at'],
    'salary_structures' => ['id', 'staff_id', 'basic_salary', 'allowance_details', 'deduction_details', 'effective_date', 'created_at'],
    'users' => ['id', 'username', 'email', 'role', 'created_at'],
    'students' => ['id', 'student_id', 'first_name', 'last_name', 'class_id', 'enrollment_date'],
    'academic_years' => ['id', 'year', 'start_date', 'end_date', 'is_active'],
    'classes' => ['id', 'class_name', 'level_id', 'academic_year_id', 'created_at'],
    'courses' => ['id', 'course_name', 'course_code', 'class_id', 'teacher_id', 'academic_year_id', 'term_id', 'created_at'],
    'grades' => ['id', 'student_id', 'course_id', 'grade', 'grade_point', 'term_id', 'academic_year_id', 'created_at'],
    'teachers' => ['id', 'staff_id', 'first_name', 'last_name', 'email', 'phone', 'subject_specialization', 'created_at'],
    'head_teachers' => ['id', 'staff_id', 'first_name', 'last_name', 'email', 'phone', 'school_section', 'created_at'],
    'non_teaching_staff' => ['id', 'staff_id', 'first_name', 'last_name', 'email', 'phone', 'position', 'department', 'created_at'],
    'directors' => ['id', 'staff_id', 'first_name', 'last_name', 'email', 'phone', 'position', 'created_at']
];

include 'includes/accounting_header.php';
?>

<style>
    .report-builder-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 0;
        background: transparent;
        min-height: auto;
        box-shadow: none;
    }

    .report-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 30px;
        border-radius: 12px;
        margin-bottom: 30px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.1);
        display: flex;
        justify-content: space-between;
        align-items: center;
    }

    .report-header h1 {
        margin: 0;
        font-size: 2.2em;
        font-weight: 300;
    }

    .report-header p {
        margin: 5px 0 0 0;
        opacity: 0.9;
        font-size: 1.1em;
    }

    .report-content {
        display: grid;
        grid-template-columns: 1fr 320px;
        gap: 30px;
    }

    .report-main {
        background: white;
        border-radius: 12px;
        box-shadow: 0 2px 15px rgba(0,0,0,0.08);
        border: 1px solid #e1e5e9;
    }

    .report-sidebar {
        background: white;
        border-radius: 12px;
        box-shadow: 0 2px 15px rgba(0,0,0,0.08);
        padding: 25px;
        border: 1px solid #e1e5e9;
    }

    .builder-section {
        padding: 25px;
    }

    .builder-section h2 {
        margin: 0 0 20px 0;
        color: #2c3e50;
        font-size: 1.5em;
        font-weight: 300;
    }

    .form-section {
        background: #f8f9fa;
        padding: 25px;
        border-radius: 12px;
        margin-bottom: 25px;
        border: 1px solid #e1e5e9;
    }

    .form-section h3 {
        margin: 0 0 20px 0;
        color: #495057;
        font-size: 1.2em;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .form-group {
        margin-bottom: 20px;
    }

    .form-group label {
        display: block;
        margin-bottom: 8px;
        font-weight: 600;
        color: #495057;
        font-size: 0.95em;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .form-group input[type="text"],
    .form-group input[type="date"],
    .form-group textarea,
    .form-group select {
        width: 100%;
        padding: 12px 16px;
        border: 2px solid #667eea;
        border-radius: 8px;
        font-size: 14px;
        transition: border-color 0.3s ease;
    }

    .form-group input:focus,
    .form-group textarea:focus,
    .form-group select:focus {
        outline: none;
        border-color: #764ba2;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }

    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }

    .table-selector {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 15px;
    }

    .table-option {
        border: 2px solid #e9ecef;
        border-radius: 8px;
        padding: 20px;
        background: #f8f9fa;
        transition: all 0.3s ease;
    }

    .table-option:hover {
        border-color: #667eea;
        background: white;
        transform: translateY(-2px);
        box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    }

    .table-option label {
        font-weight: 600;
        margin-bottom: 10px;
        display: block;
        cursor: pointer;
    }

    .column-selector {
        margin-top: 15px;
        padding-top: 15px;
        border-top: 1px solid #dee2e6;
    }

    .column-checkbox {
        display: inline-block;
        margin: 5px 10px 5px 0;
        font-weight: normal !important;
    }

    .column-checkbox input[type="checkbox"] {
        margin-right: 8px;
    }

    .filter-row,
    .column-row {
        display: grid;
        grid-template-columns: 2fr 1fr 2fr auto;
        gap: 10px;
        margin-bottom: 15px;
        align-items: end;
    }

    .filter-row select,
    .filter-row input,
    .column-row input,
    .column-row select {
        padding: 10px 12px;
        border: 2px solid #667eea;
        border-radius: 6px;
        font-size: 14px;
        transition: border-color 0.3s ease;
    }

    .filter-row select:focus,
    .filter-row input:focus,
    .column-row input:focus,
    .column-row select:focus {
        outline: none;
        border-color: #764ba2;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }

    .form-actions {
        display: flex;
        gap: 15px;
        padding-top: 25px;
        border-top: 2px solid #e9ecef;
        justify-content: center;
    }

    .btn {
        padding: 12px 24px;
        border: none;
        border-radius: 8px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        transition: all 0.3s ease;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .btn-primary {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
    }

    .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    }

    .btn-secondary {
        background: #6c757d;
        color: white;
    }

    .btn-secondary:hover {
        background: #5a6268;
        transform: translateY(-1px);
    }

    .btn-success {
        background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
        color: white;
        box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
    }

    .btn-success:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(40, 167, 69, 0.4);
    }

    .reports-list {
        display: flex;
        flex-direction: column;
        gap: 15px;
    }

    .report-card {
        border: 1px solid #e9ecef;
        border-radius: 12px;
        padding: 20px;
        background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
        transition: all 0.3s ease;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }

    .report-card:hover {
        transform: translateY(-3px);
        box-shadow: 0 4px 20px rgba(0,0,0,0.1);
        border-color: #667eea;
    }

    .report-info h3 {
        margin: 0 0 10px 0;
        color: #2c3e50;
        font-size: 1.1em;
    }

    .report-info p {
        margin: 0 0 10px 0;
        color: #7f8c8d;
        font-size: 0.9em;
    }

    .report-info small {
        color: #95a5a6;
        font-size: 0.8em;
    }

    .report-actions {
        display: flex;
        gap: 10px;
        margin-top: 15px;
    }

    .action-btn {
        padding: 8px 16px;
        border: none;
        border-radius: 6px;
        cursor: pointer;
        font-size: 0.85em;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        transition: all 0.3s ease;
    }

    .edit-btn {
        background: linear-gradient(135deg, #ffc107 0%, #fd7e14 100%);
        color: #212529;
    }

    .edit-btn:hover {
        transform: translateY(-1px);
        box-shadow: 0 4px 10px rgba(255, 193, 7, 0.3);
    }

    .delete-btn {
        background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
        color: white;
    }

    .delete-btn:hover {
        transform: translateY(-1px);
        box-shadow: 0 4px 10px rgba(220, 53, 69, 0.3);
    }

    .run-btn {
        background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
        color: white;
    }

    .run-btn:hover {
        transform: translateY(-1px);
        box-shadow: 0 4px 10px rgba(40, 167, 69, 0.3);
    }

    .no-reports {
        text-align: center;
        padding: 50px 20px;
        color: #6c757d;
    }

    .no-reports h3 {
        margin: 0 0 10px 0;
        color: #495057;
    }

    .message {
        padding: 15px;
        margin-bottom: 20px;
        border-radius: 8px;
        font-weight: 600;
    }

    .message.success {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .message.error {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .preview-section {
        margin-top: 30px;
        padding: 25px;
        background: #f8f9fa;
        border-radius: 12px;
        border: 1px solid #e1e5e9;
    }

    .preview-section h2 {
        margin: 0 0 20px 0;
        color: #2c3e50;
        font-size: 1.3em;
    }

    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }

    @media (max-width: 768px) {
        .report-content {
            grid-template-columns: 1fr;
        }

        .filter-row,
        .column-row {
            grid-template-columns: 1fr;
        }

        .report-card {
            flex-direction: column;
            gap: 15px;
        }

        .report-actions {
            justify-content: flex-start;
        }

        .form-actions {
            flex-direction: column;
        }
    }
</style>

<div class="report-builder-container">
    <div class="report-header">
        <h1>🔧 Custom Report Builder</h1>
        <p>Create custom reports with flexible data selection and filtering options</p>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="report-content">
        <div class="report-main">
            <!-- Report Builder Form -->
            <div class="builder-section">
                <h2>Create New Report</h2>
                <form method="post" class="report-form">
                    <input type="hidden" name="action" value="save_report">

                    <div class="form-section">
                        <h3>📋 Report Details</h3>
                        <div class="form-group">
                            <label for="report_name">Report Name:</label>
                            <input type="text" id="report_name" name="report_name" required>
                        </div>

                        <div class="form-group">
                            <label for="report_description">Description:</label>
                            <textarea id="report_description" name="report_description" rows="3"></textarea>
                        </div>

                        <div class="form-group">
                            <label for="report_type">Report Type:</label>
                            <select id="report_type" name="report_type" required>
                                <option value="budget_vs_actual">Budget vs Actual</option>
                                <option value="cash_flow">Cash Flow</option>
                                <option value="profit_loss">Profit & Loss</option>
                                <option value="balance_sheet">Balance Sheet</option>
                                <option value="custom">Custom Query</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label>
                                <input type="checkbox" name="is_public" value="1">
                                Make this report public (visible to all users)
                            </label>
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>📊 Data Selection</h3>
                        <div class="form-group">
                            <label>Select Tables:</label>
                            <div class="table-selector">
                                <?php foreach ($available_tables as $table_name => $columns): ?>
                                    <div class="table-option">
                                        <label>
                                            <input type="checkbox" name="tables[]" value="<?php echo $table_name; ?>"
                                                   onchange="toggleTableColumns('<?php echo $table_name; ?>')">
                                            <?php echo ucwords(str_replace('_', ' ', $table_name)); ?>
                                        </label>
                                        <div id="columns_<?php echo $table_name; ?>" class="column-selector" style="display: none;">
                                            <small>Select columns:</small>
                                            <?php foreach ($columns as $column): ?>
                                                <label class="column-checkbox">
                                                    <input type="checkbox" name="columns[<?php echo $table_name; ?>][]"
                                                           value="<?php echo $column; ?>">
                                                    <?php echo $column; ?>
                                                </label>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>🔍 Filters</h3>
                        <div id="filter-builder">
                            <div class="filter-row">
                                <select name="filters[0][column]" class="filter-column">
                                    <option value="">Select Column</option>
                                    <?php foreach ($available_tables as $table_name => $columns): ?>
                                        <optgroup label="<?php echo ucwords(str_replace('_', ' ', $table_name)); ?>">
                                            <?php foreach ($columns as $column): ?>
                                                <option value="<?php echo $table_name . '.' . $column; ?>">
                                                    <?php echo $column; ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </optgroup>
                                    <?php endforeach; ?>
                                </select>
                                <select name="filters[0][operator]" class="filter-operator">
                                    <option value="=">Equals</option>
                                    <option value="!=">Not Equals</option>
                                    <option value=">">Greater Than</option>
                                    <option value="<">Less Than</option>
                                    <option value=">=">Greater or Equal</option>
                                    <option value="<=">Less or Equal</option>
                                    <option value="LIKE">Contains</option>
                                    <option value="IN">In</option>
                                </select>
                                <input type="text" name="filters[0][value]" class="filter-value" placeholder="Value">
                                <button type="button" onclick="addFilter()">Add Filter</button>
                            </div>
                        </div>
                    </div>

                    <div class="form-section">
                        <h3>📈 Column Configuration</h3>
                        <div id="column-config">
                            <div class="column-row">
                                <input type="text" name="columns_config[0][name]" placeholder="Column Name" class="column-name">
                                <input type="text" name="columns_config[0][expression]" placeholder="SQL Expression" class="column-expression">
                                <select name="columns_config[0][format]" class="column-format">
                                    <option value="text">Text</option>
                                    <option value="number">Number</option>
                                    <option value="currency">Currency</option>
                                    <option value="date">Date</option>
                                    <option value="percentage">Percentage</option>
                                </select>
                                <button type="button" onclick="addColumn()">Add Column</button>
                            </div>
                        </div>
                    </div>

                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary">💾 Save Report</button>
                        <button type="button" onclick="previewReport()" class="btn btn-secondary">👁️ Preview</button>
                    </div>
                </form>
            </div>

            <!-- Report Preview -->
            <div class="preview-section" id="preview-section" style="display: none;">
                <h2>Report Preview</h2>
                <div id="report-preview" class="table-container">
                    <!-- Preview results will be loaded here -->
                </div>
            </div>
        </div>

        <div class="report-sidebar">
            <h3 style="margin-top: 0; color: #2c3e50;">📁 Saved Reports</h3>
            <div class="reports-list">
                <?php if (empty($saved_reports)): ?>
                    <div class="no-reports">
                        <h3>No reports yet</h3>
                        <p>Create your first custom report using the form.</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($saved_reports as $report): ?>
                        <div class="report-card">
                            <div class="report-info">
                                <h3><?php echo htmlspecialchars($report['name']); ?></h3>
                                <p><?php echo htmlspecialchars($report['description']); ?></p>
                                <small>
                                    Type: <?php echo ucwords(str_replace('_', ' ', $report['report_type'])); ?> |
                                    Created by: <?php echo htmlspecialchars($report['created_by_name']); ?> |
                                    <?php echo $report['is_public'] ? 'Public' : 'Private'; ?>
                                </small>
                            </div>
                            <div class="report-actions">
                                <form method="post" style="display: inline;">
                                    <input type="hidden" name="action" value="run_report">
                                    <input type="hidden" name="report_id" value="<?php echo $report['id']; ?>">
                                    <button type="button" class="action-btn run-btn" onclick="runReport(<?php echo $report['id']; ?>)">▶️ Run</button>
                                </form>
                                <button type="button" class="action-btn edit-btn" onclick="editReport(<?php echo $report['id']; ?>)">✏️ Edit</button>
                                <button type="button" class="action-btn delete-btn" onclick="deleteReport(<?php echo $report['id']; ?>)">🗑️ Delete</button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>

            <h3 style="margin-top: 30px; color: #2c3e50;">🔗 Quick Actions</h3>
            <div style="display: flex; flex-direction: column; gap: 10px;">
                <a href="index.php" class="btn btn-secondary" style="text-align: center;">🏠 Dashboard</a>
                <a href="expense_list.php" class="btn btn-primary" style="text-align: center;">💰 Expenses</a>
                <a href="budget_vs_actual_report.php" class="btn btn-primary" style="text-align: center;">📊 Budget Reports</a>
            </div>
        </div>
    </div>
</div>

<style>
.custom-report-builder {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.report-header {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-header h1 {
    color: #2c3e50;
    margin: 0 0 10px 0;
}

.report-header p {
    color: #7f8c8d;
    margin: 0;
}

.report-builder-container {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 30px;
}

.builder-section,
.saved-reports-section,
.preview-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.builder-section h2,
.saved-reports-section h2,
.preview-section h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    border-bottom: 2px solid #e9ecef;
    padding-bottom: 10px;
}

.form-section {
    margin-bottom: 30px;
}

.form-section h3 {
    color: #34495e;
    margin-bottom: 15px;
    padding: 10px 15px;
    background: #f8f9fa;
    border-radius: 5px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 5px;
    font-weight: 600;
    color: #495057;
}

.form-group input[type="text"],
.form-group input[type="date"],
.form-group textarea,
.form-group select {
    width: 100%;
    padding: 10px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.form-group textarea {
    resize: vertical;
    min-height: 80px;
}

.table-selector {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
}

.table-option {
    border: 1px solid #e9ecef;
    border-radius: 5px;
    padding: 15px;
    background: #f8f9fa;
}

.table-option label {
    font-weight: 600;
    margin-bottom: 10px;
    display: block;
}

.column-selector {
    margin-top: 10px;
    padding-top: 10px;
    border-top: 1px solid #dee2e6;
}

.column-checkbox {
    display: inline-block;
    margin: 5px 10px 5px 0;
    font-weight: normal !important;
}

.column-checkbox input[type="checkbox"] {
    margin-right: 5px;
}

.filter-row,
.column-row {
    display: grid;
    grid-template-columns: 2fr 1fr 2fr auto;
    gap: 10px;
    margin-bottom: 10px;
    align-items: center;
}

.filter-row select,
.filter-row input,
.column-row input,
.column-row select {
    padding: 8px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.form-actions {
    display: flex;
    gap: 15px;
    padding-top: 20px;
    border-top: 2px solid #e9ecef;
}

.reports-list {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.report-card {
    border: 1px solid #e9ecef;
    border-radius: 8px;
    padding: 20px;
    background: #f8f9fa;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
}

.report-info h3 {
    margin: 0 0 10px 0;
    color: #2c3e50;
}

.report-info p {
    margin: 0 0 10px 0;
    color: #7f8c8d;
}

.report-info small {
    color: #95a5a6;
}

.report-actions {
    display: flex;
    gap: 10px;
}

.no-reports {
    text-align: center;
    color: #7f8c8d;
    font-style: italic;
    padding: 40px;
}

.table-container {
    overflow-x: auto;
    margin-top: 20px;
}

.success-message {
    background: #d4edda;
    color: #155724;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #c3e6cb;
}

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

.btn {
    padding: 10px 20px;
    border: none;
    border-radius: 4px;
    cursor: pointer;
    font-size: 14px;
    text-decoration: none;
    display: inline-block;
    text-align: center;
    transition: background-color 0.2s;
}

.btn-primary {
    background: #007bff;
    color: white;
}

.btn-primary:hover {
    background: #0056b3;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #545b62;
}

.btn-danger {
    background: #dc3545;
    color: white;
}

.btn-danger:hover {
    background: #c82333;
}

.btn-small {
    padding: 5px 10px;
    font-size: 12px;
}

@media (max-width: 768px) {
    .report-builder-container {
        grid-template-columns: 1fr;
    }

    .filter-row,
    .column-row {
        grid-template-columns: 1fr;
    }

    .report-card {
        flex-direction: column;
        gap: 15px;
    }

    .report-actions {
        justify-content: flex-start;
    }
}
</style>

<script>
let filterCount = 1;
let columnCount = 1;

function toggleTableColumns(tableName) {
    const columnSelector = document.getElementById('columns_' + tableName);
    const checkbox = document.querySelector(`input[name="tables[]"][value="${tableName}"]`);

    if (checkbox.checked) {
        columnSelector.style.display = 'block';
    } else {
        columnSelector.style.display = 'none';
    }
}

function addFilter() {
    const filterBuilder = document.getElementById('filter-builder');
    const newFilterRow = document.createElement('div');
    newFilterRow.className = 'filter-row';
    newFilterRow.innerHTML = `
        <select name="filters[${filterCount}][column]" class="filter-column">
            <option value="">Select Column</option>
            <?php foreach ($available_tables as $table_name => $columns): ?>
                <optgroup label="<?php echo ucwords(str_replace('_', ' ', $table_name)); ?>">
                    <?php foreach ($columns as $column): ?>
                        <option value="<?php echo $table_name . '.' . $column; ?>">
                            <?php echo $column; ?>
                        </option>
                    <?php endforeach; ?>
                </optgroup>
            <?php endforeach; ?>
        </select>
        <select name="filters[${filterCount}][operator]" class="filter-operator">
            <option value="=">Equals</option>
            <option value="!=">Not Equals</option>
            <option value=">">Greater Than</option>
            <option value="<">Less Than</option>
            <option value=">=">Greater or Equal</option>
            <option value="<=">Less or Equal</option>
            <option value="LIKE">Contains</option>
            <option value="IN">In</option>
        </select>
        <input type="text" name="filters[${filterCount}][value]" class="filter-value" placeholder="Value">
        <button type="button" onclick="removeFilter(this)">Remove</button>
    `;
    filterBuilder.appendChild(newFilterRow);
    filterCount++;
}

function removeFilter(button) {
    button.parentElement.remove();
}

function addColumn() {
    const columnConfig = document.getElementById('column-config');
    const newColumnRow = document.createElement('div');
    newColumnRow.className = 'column-row';
    newColumnRow.innerHTML = `
        <input type="text" name="columns_config[${columnCount}][name]" placeholder="Column Name" class="column-name">
        <input type="text" name="columns_config[${columnCount}][expression]" placeholder="SQL Expression" class="column-expression">
        <select name="columns_config[${columnCount}][format]" class="column-format">
            <option value="text">Text</option>
            <option value="number">Number</option>
            <option value="currency">Currency</option>
            <option value="date">Date</option>
            <option value="percentage">Percentage</option>
        </select>
        <button type="button" onclick="removeColumn(this)">Remove</button>
    `;
    columnConfig.appendChild(newColumnRow);
    columnCount++;
}

function removeColumn(button) {
    button.parentElement.remove();
}

function previewReport() {
    // Show preview section
    document.getElementById('preview-section').style.display = 'block';

    // In a real implementation, this would make an AJAX call to preview the report
    document.getElementById('report-preview').innerHTML = `
        <p>Report preview functionality would be implemented here.</p>
        <p>This would show a sample of the data based on your current configuration.</p>
    `;
}

function editReport(reportId) {
    // In a real implementation, this would load the report configuration
    alert('Edit functionality would be implemented here for report ID: ' + reportId);
}

function deleteReport(reportId) {
    if (confirm('Are you sure you want to delete this report?')) {
        // In a real implementation, this would make an AJAX call to delete the report
        alert('Delete functionality would be implemented here for report ID: ' + reportId);
    }
}

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    // Add event listeners for form validation
    document.querySelector('.report-form').addEventListener('submit', function(e) {
        const reportName = document.getElementById('report_name').value.trim();
        if (!reportName) {
            e.preventDefault();
            alert('Please enter a report name');
            return;
        }
    });
});
</script>

<?php
include 'includes/accounting_footer.php';

// Helper function to build custom query
function buildCustomQuery($query_config, $filters, $columns_config) {
    // This is a simplified example - in reality, you'd need more sophisticated query building
    $query = "SELECT ";

    if (!empty($columns_config)) {
        $select_columns = [];
        foreach ($columns_config as $column) {
            if (!empty($column['expression'])) {
                $select_columns[] = $column['expression'] . " AS " . $column['name'];
            }
        }
        $query .= implode(', ', $select_columns);
    } else {
        $query .= "*";
    }

    $query .= " FROM student_fees sf LEFT JOIN expenses e ON 1=1";

    // Add WHERE clause for filters
    if (!empty($filters)) {
        $where_conditions = [];
        foreach ($filters as $filter) {
            if (!empty($filter['column']) && !empty($filter['value'])) {
                $where_conditions[] = $filter['column'] . " " . $filter['operator'] . " '" . addslashes($filter['value']) . "'";
            }
        }
        if (!empty($where_conditions)) {
            $query .= " WHERE " . implode(' AND ', $where_conditions);
        }
    }

    $query .= " LIMIT 100"; // Limit for preview

    return $query;
}
?>
