<?php
/**
 * Create journal entry for asset acquisition (accrual basis)
 * Debit: Property, Plant & Equipment
 * Credit: Cash Account (based on payment method) or Accounts Payable
 */

require_once 'accounting_functions.php';
function createAssetAcquisitionEntry($pdo, $assetId, $amount, $paymentMethod, $date, $createdBy, $manageTransaction = true) {
    try {
        if ($manageTransaction) {
            $pdo->beginTransaction();
        }

        // Get asset details including category for account mapping
        $stmt = $pdo->prepare("SELECT a.name, a.asset_type, ac.name as category_name FROM assets a LEFT JOIN asset_categories ac ON a.category_id = ac.id WHERE a.id = ?");
        $stmt->execute([$assetId]);
        $asset = $stmt->fetch();

        if (!$asset) {
            throw new Exception('Asset not found');
        }

        // Map asset categories to fixed asset accounts
        $categoryMapping = [
            1 => 'Buildings',                    // Buildings
            2 => 'Vehicles',                     // Vehicles
            3 => 'Furniture & Equipment',        // Furniture and Fixtures
            4 => 'Computers and IT Equipment',   // Computers and IT Equipment
            5 => 'Furniture & Equipment',        // Office Equipment
            6 => 'Furniture & Equipment',       // Machinery and Equipment
            7 => 'Land',                         // Land
            8 => 'Computers and IT Equipment',   // Software
            9 => 'Buildings',                    // Leasehold Improvements
            10 => 'Buildings',                   // Intangible Assets (using Buildings as default)
            11 => 'Furniture & Equipment',       // Laboratory Equipment
            12 => 'Furniture & Equipment',       // Library Books
            13 => 'Buildings'                    // Construction in Progress
        ];

        // Get category_id from assets table
        $stmt = $pdo->prepare("SELECT category_id FROM assets WHERE id = ?");
        $stmt->execute([$assetId]);
        $categoryResult = $stmt->fetch();
        $categoryId = $categoryResult['category_id'];

        $accountName = $categoryMapping[$categoryId] ?? 'Furniture & Equipment';
        $ppeAccountId = getChartAccountId($pdo, $accountName);
        if (!$ppeAccountId) {
            throw new Exception("Account '{$accountName}' not found for asset type '{$asset['asset_type']}'");
        }

        // Get cash account based on payment method
        $cashAccountName = PaymentMethods::getCashAccountForPaymentMethod($paymentMethod);
        if ($paymentMethod === 'donated') {
            // For donated assets, create a journal entry that credits Donations (increases revenue/equity)
            // Debit: Property, Plant & Equipment (increases asset)
            // Credit: Donations (increases revenue/equity)

            $entryNumber = generateEntryNumber($pdo, $date);

            // Create journal entry
            $stmt = $pdo->prepare("
                INSERT INTO journal_entries
                (entry_number, transaction_date, description, source_type, source_id, status, created_by)
                VALUES (?, ?, ?, 'asset', ?, 'posted', ?)
            ");
            $stmt->execute([
                $entryNumber,
                $date,
                "Donated asset: {$asset['name']}",
                $assetId,
                $createdBy
            ]);
            $entryId = $pdo->lastInsertId();

            // Debit: Property, Plant & Equipment (increases asset)
            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, debit_amount, line_number, description)
                VALUES (?, ?, ?, 1, ?)
            ");
            $stmt->execute([
                $entryId,
                $ppeAccountId,
                $amount,
                "Donated asset: {$asset['name']}"
            ]);

            // Credit: Capital Fund (increases equity directly)
            $equityId = getChartAccountId($pdo, 'Capital Fund');
            if (!$equityId) {
                throw new Exception('Capital Fund account not found');
            }

            $stmt = $pdo->prepare("
                INSERT INTO journal_entry_lines
                (journal_entry_id, account_id, credit_amount, line_number, description)
                VALUES (?, ?, ?, 2, ?)
            ");
            $stmt->execute([
                $entryId,
                $equityId,
                $amount,
                "Donation for asset: {$asset['name']}"
            ]);

            // Update account balances
            $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
            $stmt->execute([$amount, $ppeAccountId]); // Debit increases asset

            $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
            $stmt->execute([$amount, $equityId]); // Credit increases equity

            if ($manageTransaction) {
                $pdo->commit();
            }
            return $entryId;
        }

        $cashAccountId = getChartAccountId($pdo, $cashAccountName);
        if (!$cashAccountId) {
            // If specific cash account not found, try Accounts Payable for credit purchases
            $cashAccountId = getChartAccountId($pdo, 'Accounts Payable');
            if (!$cashAccountId) {
                throw new Exception('Neither cash account nor Accounts Payable found');
            }
        }

        $entryNumber = generateEntryNumber($pdo, $date);

        // Create journal entry
        $stmt = $pdo->prepare("
            INSERT INTO journal_entries
            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
            VALUES (?, ?, ?, 'asset', ?, 'posted', ?)
        ");
        $stmt->execute([
            $entryNumber,
            $date,
            "Asset acquisition: {$asset['name']}",
            $assetId,
            $createdBy
        ]);
        $entryId = $pdo->lastInsertId();

        // Debit: Property, Plant & Equipment (increases asset)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, debit_amount, line_number, description)
            VALUES (?, ?, ?, 1, ?)
        ");
        $stmt->execute([
            $entryId,
            $ppeAccountId,
            $amount,
            "Asset acquisition: {$asset['name']}"
        ]);

        // Credit: Cash Account or Accounts Payable (decreases cash/liability)
        $stmt = $pdo->prepare("
            INSERT INTO journal_entry_lines
            (journal_entry_id, account_id, credit_amount, line_number, description)
            VALUES (?, ?, ?, 2, ?)
        ");
        $stmt->execute([
            $entryId,
            $cashAccountId,
            $amount,
            "Payment for asset: {$asset['name']} ({$cashAccountName})"
        ]);

        // Update account balances
        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
        $stmt->execute([$amount, $ppeAccountId]); // Debit increases asset

        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
        $stmt->execute([$amount, $cashAccountId]); // Credit decreases asset/liability

        if ($manageTransaction) {
            $pdo->commit();
        }
        return $entryId;

    } catch (Exception $e) {
        if ($manageTransaction) {
            $pdo->rollBack();
        }
        throw $e;
    }
}
?>
