<?php
require_once '../config.php';

$message = '';
$error = '';

// Get filter parameters
$account_id = $_GET['account_id'] ?? '';
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');
$transaction_type = $_GET['transaction_type'] ?? 'all';

// Fetch payment method accounts for dropdown (from chart of accounts)
try {
    $stmt = $pdo->query("SELECT id, account_code, account_name FROM chart_of_accounts WHERE account_code IN ('1001', '1120', '1124') ORDER BY account_name ASC");
    $cash_accounts = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading payment method accounts: " . $e->getMessage();
    $cash_accounts = [];
}

// Build cashbook query
try {
    $where_conditions = [];
    $params = [];

    // Date range
    $where_conditions[] = "DATE(ct.created_at) BETWEEN ? AND ?";
    $params[] = $start_date;
    $params[] = $end_date;

    // Account filter
    if (!empty($account_id)) {
        $where_conditions[] = "ct.account_id = ?";
        $params[] = $account_id;
    }

    // Transaction type filter
    if ($transaction_type !== 'all') {
        $where_conditions[] = "ct.transaction_type = ?";
        $params[] = $transaction_type;
    }

    $where_clause = implode(' AND ', $where_conditions);

    // Get cashbook entries (excluding payment method accounts)
    $query = "
        SELECT
            ct.id,
            ct.created_at as transaction_date,
            ca.account_name,
            ca.account_type,
            ct.transaction_type,
            ct.amount,
            ct.description,
            ct.created_by
        FROM cash_account_transactions ct
        JOIN cash_accounts ca ON ct.account_id = ca.id
        LEFT JOIN chart_of_accounts coa ON ca.account_code = coa.account_code
        WHERE {$where_clause}
        AND (coa.account_code IS NULL OR coa.account_code NOT IN ('1001', '1120', '1124'))
        ORDER BY ct.created_at ASC, ct.id ASC
    ";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $cashbook_entries = $stmt->fetchAll();

    // Calculate running balances
    $running_balance = 0;
    $processed_entries = [];

    foreach ($cashbook_entries as $entry) {
        if ($entry['transaction_type'] === 'increase') {
            $running_balance += $entry['amount'];
            $debit = $entry['amount'];
            $credit = 0;
        } else {
            $running_balance -= $entry['amount'];
            $debit = 0;
            $credit = $entry['amount'];
        }

        $processed_entries[] = [
            'date' => $entry['transaction_date'],
            'account' => $entry['account_name'],
            'description' => $entry['description'],
            'debit' => $debit,
            'credit' => $credit,
            'balance' => $running_balance,
            'type' => $entry['transaction_type']
        ];
    }

    // Get summary statistics
    $total_debits = array_sum(array_column($processed_entries, 'debit'));
    $total_credits = array_sum(array_column($processed_entries, 'credit'));
    $net_movement = $total_debits - $total_credits;

    // Get opening balance (balance before start date)
    if (!empty($account_id)) {
        $balance_query = "
            SELECT COALESCE(SUM(CASE WHEN transaction_type = 'increase' THEN amount ELSE -amount END), 0) as opening_balance
            FROM cash_account_transactions
            WHERE account_id = ? AND DATE(created_at) < ?
        ";
        $stmt = $pdo->prepare($balance_query);
        $stmt->execute([$account_id, $start_date]);
        $opening_balance = $stmt->fetch()['opening_balance'];
    } else {
        // For all accounts combined
        $balance_query = "
            SELECT COALESCE(SUM(CASE WHEN transaction_type = 'increase' THEN amount ELSE -amount END), 0) as opening_balance
            FROM cash_account_transactions
            WHERE DATE(created_at) < ?
        ";
        $stmt = $pdo->prepare($balance_query);
        $stmt->execute([$start_date]);
        $opening_balance = $stmt->fetch()['opening_balance'];
    }

    $closing_balance = $opening_balance + $net_movement;

} catch (PDOException $e) {
    $error = "Error loading cashbook data: " . $e->getMessage();
    $processed_entries = [];
    $total_debits = $total_credits = $net_movement = $opening_balance = $closing_balance = 0;
}

include 'includes/accounting_header.php';
?>

<div class="cashbook-container">
    <div class="page-header">
        <h1>📖 Cash Book</h1>
        <p>Chronological record of all cash transactions</p>
    </div>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Filters -->
    <div class="filters-section">
        <form method="get" class="filters-form">
            <div class="filter-row">
                <div class="filter-group">
                    <label for="account_id">Account:</label>
                    <select name="account_id" id="account_id">
                        <option value="">All Accounts</option>
                        <?php foreach ($cash_accounts as $account): ?>
                            <option value="<?php echo $account['id']; ?>" <?php echo $account_id == $account['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($account['account_name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="start_date">Start Date:</label>
                    <input type="date" name="start_date" id="start_date" value="<?php echo $start_date; ?>">
                </div>

                <div class="filter-group">
                    <label for="end_date">End Date:</label>
                    <input type="date" name="end_date" id="end_date" value="<?php echo $end_date; ?>">
                </div>

                <div class="filter-group">
                    <label for="transaction_type">Type:</label>
                    <select name="transaction_type" id="transaction_type">
                        <option value="all" <?php echo $transaction_type === 'all' ? 'selected' : ''; ?>>All Transactions</option>
                        <option value="increase" <?php echo $transaction_type === 'increase' ? 'selected' : ''; ?>>Cash In</option>
                        <option value="decrease" <?php echo $transaction_type === 'decrease' ? 'selected' : ''; ?>>Cash Out</option>
                    </select>
                </div>

                <div class="filter-group">
                    <button type="submit" class="btn btn-primary">Filter</button>
                    <a href="cashbook.php" class="btn btn-secondary">Reset</a>
                </div>
            </div>
        </form>
    </div>

    <!-- Summary Cards -->
    <div class="summary-cards">
        <div class="summary-card">
            <h4>Opening Balance</h4>
            <p class="amount">GHC <?php echo number_format($opening_balance, 2); ?></p>
        </div>

        <div class="summary-card">
            <h4>Total Cash In</h4>
            <p class="amount positive">GHC <?php echo number_format($total_debits, 2); ?></p>
        </div>

        <div class="summary-card">
            <h4>Total Cash Out</h4>
            <p class="amount negative">GHC <?php echo number_format($total_credits, 2); ?></p>
        </div>

        <div class="summary-card">
            <h4>Net Movement</h4>
            <p class="amount <?php echo $net_movement >= 0 ? 'positive' : 'negative'; ?>">
                GHC <?php echo number_format($net_movement, 2); ?>
            </p>
        </div>

        <div class="summary-card">
            <h4>Closing Balance</h4>
            <p class="amount <?php echo $closing_balance >= 0 ? 'positive' : 'negative'; ?>">
                GHC <?php echo number_format($closing_balance, 2); ?>
            </p>
        </div>
    </div>

    <!-- Cash Book Table -->
    <div class="cashbook-table-container">
        <div class="table-header">
            <h3>Cash Book Entries</h3>
            <div class="table-actions">
                <button onclick="window.print()" class="btn btn-secondary">Print</button>
                <a href="export_cashbook.php?<?php echo http_build_query($_GET); ?>" class="btn btn-secondary">Export</a>
            </div>
        </div>

        <?php if (empty($processed_entries)): ?>
            <div class="no-entries">
                <p>No cash transactions found for the selected criteria.</p>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="cashbook-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Account</th>
                            <th>Description</th>
                            <th>Cash In (GHC)</th>
                            <th>Cash Out (GHC)</th>
                            <th>Balance (GHC)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- Opening Balance Row -->
                        <tr class="opening-balance-row">
                            <td><?php echo date('M d, Y', strtotime($start_date)); ?></td>
                            <td colspan="4"><strong>Opening Balance</strong></td>
                            <td class="balance-cell"><strong><?php echo number_format($opening_balance, 2); ?></strong></td>
                        </tr>

                        <?php foreach ($processed_entries as $entry): ?>
                            <tr class="<?php echo $entry['type'] === 'increase' ? 'cash-in-row' : 'cash-out-row'; ?>">
                                <td><?php echo date('M d, Y H:i', strtotime($entry['date'])); ?></td>
                                <td><?php echo htmlspecialchars($entry['account']); ?></td>
                                <td><?php echo htmlspecialchars($entry['description']); ?></td>
                                <td class="debit-cell"><?php echo $entry['debit'] > 0 ? number_format($entry['debit'], 2) : '-'; ?></td>
                                <td class="credit-cell"><?php echo $entry['credit'] > 0 ? number_format($entry['credit'], 2) : '-'; ?></td>
                                <td class="balance-cell"><?php echo number_format($entry['balance'], 2); ?></td>
                            </tr>
                        <?php endforeach; ?>

                        <!-- Closing Balance Row -->
                        <tr class="closing-balance-row">
                            <td><?php echo date('M d, Y', strtotime($end_date)); ?></td>
                            <td colspan="4"><strong>Closing Balance</strong></td>
                            <td class="balance-cell"><strong><?php echo number_format($closing_balance, 2); ?></strong></td>
                        </tr>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
.cashbook-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.page-header {
    background: white;
    padding: 30px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.page-header h1 {
    color: #2c3e50;
    margin: 0 0 10px 0;
}

.page-header p {
    color: #7f8c8d;
    margin: 0;
}

.filters-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.filters-form {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.filter-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group select,
.filter-group input {
    padding: 10px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.summary-card {
    background: white;
    padding: 20px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.summary-card h4 {
    color: #2c3e50;
    margin: 0 0 10px 0;
    font-size: 0.9em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.summary-card .amount {
    font-size: 1.5em;
    font-weight: 600;
    margin: 0;
}

.summary-card .amount.positive {
    color: #27ae60;
}

.summary-card .amount.negative {
    color: #e74c3c;
}

.cashbook-table-container {
    background: white;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    overflow: hidden;
}

.table-header {
    padding: 20px 25px;
    border-bottom: 1px solid #e9ecef;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.table-header h3 {
    color: #2c3e50;
    margin: 0;
}

.table-actions {
    display: flex;
    gap: 10px;
}

.btn {
    padding: 8px 16px;
    border: none;
    border-radius: 4px;
    cursor: pointer;
    font-size: 14px;
    text-decoration: none;
    display: inline-block;
    transition: background-color 0.2s;
}

.btn-primary {
    background: #007bff;
    color: white;
}

.btn-primary:hover {
    background: #0056b3;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #545b62;
}

.table-responsive {
    overflow-x: auto;
}

.cashbook-table {
    width: 100%;
    border-collapse: collapse;
}

.cashbook-table th,
.cashbook-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.cashbook-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
    position: sticky;
    top: 0;
    z-index: 10;
}

.cashbook-table tr:hover {
    background: #f8f9fa;
}

.debit-cell {
    color: #27ae60;
    font-weight: 600;
}

.credit-cell {
    color: #e74c3c;
    font-weight: 600;
}

.balance-cell {
    font-weight: 600;
    color: #2c3e50;
}

.opening-balance-row,
.closing-balance-row {
    background: #e8f5e8;
    font-weight: 600;
}

.opening-balance-row td,
.closing-balance-row td {
    border-top: 2px solid #27ae60;
}

.cash-in-row {
    background: rgba(39, 174, 96, 0.05);
}

.cash-out-row {
    background: rgba(231, 76, 60, 0.05);
}

.no-entries {
    padding: 40px;
    text-align: center;
    color: #7f8c8d;
}

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

@media (max-width: 768px) {
    .filter-row {
        grid-template-columns: 1fr;
    }

    .summary-cards {
        grid-template-columns: 1fr;
    }

    .table-header {
        flex-direction: column;
        gap: 15px;
        text-align: center;
    }

    .cashbook-table th,
    .cashbook-table td {
        padding: 8px 10px;
        font-size: 14px;
    }
}

@media print {
    .filters-section,
    .table-actions,
    .btn {
        display: none !important;
    }

    .cashbook-container {
        padding: 0;
        background: white;
    }

    .page-header {
        margin-bottom: 20px;
    }

    .summary-cards {
        margin-bottom: 20px;
    }
}
</style>

<?php include '../includes/admin_footer.php'; ?>
