<?php
require_once dirname(__DIR__) . '/config.php';

$message = '';
$error = '';

// Initialize variables to prevent undefined variable errors
$total_assets = 0;
$total_liabilities = 0;
$total_equity = 0;
$balance_check = 0;
$asset_accounts = [];
$liability_accounts = [];
$equity_accounts = [];
$current_assets = [];
$non_current_assets = [];
$current_liabilities = [];
$non_current_liabilities = [];
$asset_breakdown = [];
$liability_breakdown = [];
$balance_trends = [];

// Get filter parameters
$as_of_date = $_GET['as_of_date'] ?? date('Y-m-d');
$report_type = $_GET['report_type'] ?? 'current'; // current, monthly, yearly
$selected_account_types = $_GET['account_types'] ?? ['asset', 'liability', 'equity']; // Default to all types
if (!is_array($selected_account_types)) {
    $selected_account_types = [$selected_account_types];
}

// Calculate date based on report type
if ($report_type === 'yearly') {
    $as_of_date = date('Y-12-31');
} elseif ($report_type === 'monthly') {
    $as_of_date = date('Y-m-t');
}

// Get balance sheet data using double-entry bookkeeping
try {
    // Get asset balances calculated from journal entries with parent relationships
    $stmt = $pdo->prepare("
        SELECT
            coa.id,
            coa.account_code,
            coa.account_name,
            coa.account_type,
            coa.normal_balance,
            coa.parent_account_id,
            parent.account_name as parent_name,
            COALESCE(
                SUM(
                    CASE
                        WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                        ELSE jel.credit_amount - jel.debit_amount
                    END
                ), 0
            ) as balance
        FROM chart_of_accounts coa
        LEFT JOIN chart_of_accounts parent ON coa.parent_account_id = parent.id
        LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
        LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
        WHERE coa.account_type = 'asset'
        AND coa.is_active = TRUE
        GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type, coa.normal_balance, coa.parent_account_id, parent.account_name
        ORDER BY coa.account_code
    ");
    $stmt->execute([$as_of_date]);
    $raw_asset_accounts = $stmt->fetchAll();

    // Aggregate child accounts under parent accounts
    $asset_accounts = [];
    $parent_totals = [];

    foreach ($raw_asset_accounts as $account) {
        if ($account['parent_account_id']) {
            // This is a child account - add to parent's total
            $parent_id = $account['parent_account_id'];
            if (!isset($parent_totals[$parent_id])) {
                $parent_totals[$parent_id] = [
                    'account_code' => '', // Will be set when we find the parent
                    'account_name' => $account['parent_name'],
                    'account_type' => $account['account_type'],
                    'normal_balance' => $account['normal_balance'],
                    'balance' => 0,
                    'is_aggregate' => true
                ];
            }
            $parent_totals[$parent_id]['balance'] += $account['balance'];
        } else {
            // This is a top-level account - check if it has children
            $has_children = false;
            foreach ($raw_asset_accounts as $check_account) {
                if ($check_account['parent_account_id'] == $account['id']) {
                    $has_children = true;
                    break;
                }
            }

            if ($has_children) {
                // Parent account with children - will be handled by parent_totals
                continue;
            } else {
                // Standalone account
                $asset_accounts[] = $account;
            }
        }
    }

    // Add aggregated parent accounts
    foreach ($parent_totals as $parent_id => $parent_data) {
        // Find the parent's own data
        $parent_own_balance = 0;
        $parent_account_code = '';
        foreach ($raw_asset_accounts as $account) {
            if ($account['id'] == $parent_id) {
                $parent_own_balance = $account['balance'];
                $parent_account_code = $account['account_code'];
                break;
            }
        }
        // Add parent's own balance to the aggregated children
        $parent_data['balance'] += $parent_own_balance;
        $parent_data['account_code'] = $parent_account_code;
        $asset_accounts[] = $parent_data;
    }

    // Sort by account code
    usort($asset_accounts, function($a, $b) {
        return strcmp($a['account_code'], $b['account_code']);
    });

    // Get all liability accounts with their balances calculated from journal entries
    $stmt = $pdo->prepare("
        SELECT
            coa.id,
            coa.account_code,
            coa.account_name,
            coa.account_type,
            coa.normal_balance,
            coa.category,
            COALESCE(
                SUM(
                    CASE
                        WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                        ELSE jel.credit_amount - jel.debit_amount
                    END
                ), 0
            ) as balance
        FROM chart_of_accounts coa
        LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
        LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
        WHERE coa.account_type = 'liability'
        AND coa.is_active = TRUE
        GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type, coa.normal_balance, coa.category
        ORDER BY coa.account_code
    ");
    $stmt->execute([$as_of_date]);
    $liability_accounts = $stmt->fetchAll();

    // Get equity balances calculated from journal entries with parent relationships
    $stmt = $pdo->prepare("
        SELECT
            coa.id,
            coa.account_code,
            coa.account_name,
            coa.account_type,
            coa.normal_balance,
            coa.parent_account_id,
            parent.account_name as parent_name,
            COALESCE(
                SUM(
                    CASE
                        WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                        ELSE jel.credit_amount - jel.debit_amount
                    END
                ), 0
            ) as balance
        FROM chart_of_accounts coa
        LEFT JOIN chart_of_accounts parent ON coa.parent_account_id = parent.id
        LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
        LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
        WHERE coa.account_type = 'equity'
        AND coa.is_active = TRUE
        GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type, coa.normal_balance, coa.parent_account_id, parent.account_name
        ORDER BY coa.account_code
    ");
    $stmt->execute([$as_of_date]);
    $raw_equity_accounts = $stmt->fetchAll();

    // Aggregate child accounts under parent accounts for equity
    $equity_accounts = [];
    $equity_parent_totals = [];

    foreach ($raw_equity_accounts as $account) {
        if ($account['parent_account_id']) {
            // This is a child account - add to parent's total
            $parent_id = $account['parent_account_id'];
            if (!isset($equity_parent_totals[$parent_id])) {
                $equity_parent_totals[$parent_id] = [
                    'account_code' => '', // Will be set when we find the parent
                    'account_name' => $account['parent_name'],
                    'account_type' => $account['account_type'],
                    'normal_balance' => $account['normal_balance'],
                    'balance' => 0,
                    'is_aggregate' => true
                ];
            }
            $equity_parent_totals[$parent_id]['balance'] += $account['balance'];
        } else {
            // This is a top-level account - check if it has children
            $has_children = false;
            foreach ($raw_equity_accounts as $check_account) {
                if ($check_account['parent_account_id'] == $account['id']) {
                    $has_children = true;
                    break;
                }
            }

            if ($has_children) {
                // Parent account with children - will be handled by parent_totals
                continue;
            } else {
                // Standalone account
                $equity_accounts[] = $account;
            }
        }
    }

    // Add aggregated parent accounts
    foreach ($equity_parent_totals as $parent_id => $parent_data) {
        // Find the parent's own data
        $parent_own_balance = 0;
        $parent_account_code = '';
        foreach ($raw_equity_accounts as $account) {
            if ($account['id'] == $parent_id) {
                $parent_own_balance = $account['balance'];
                $parent_account_code = $account['account_code'];
                break;
            }
        }
        // Add parent's own balance to the aggregated children
        $parent_data['balance'] += $parent_own_balance;
        $parent_data['account_code'] = $parent_account_code;
        $equity_accounts[] = $parent_data;
    }

    // Sort by account code
    usort($equity_accounts, function($a, $b) {
        return strcmp($a['account_code'], $b['account_code']);
    });

    // Get revenue balances for net income calculation
    $stmt = $pdo->prepare("
        SELECT
            COALESCE(
                SUM(
                    CASE
                        WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                        ELSE jel.credit_amount - jel.debit_amount
                    END
                ), 0
            ) as total_revenue
        FROM chart_of_accounts coa
        LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
        LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
        WHERE coa.account_type = 'revenue'
        AND coa.account_code LIKE '4%'
    ");
    $stmt->execute([$as_of_date]);
    $revenue_result = $stmt->fetch();
    $total_revenue = $revenue_result['total_revenue'];

    // Get expense balances for net income calculation
    $stmt = $pdo->prepare("
        SELECT
            COALESCE(
                SUM(
                    CASE
                        WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                        ELSE jel.credit_amount - jel.debit_amount
                    END
                ), 0
            ) as total_expense
        FROM chart_of_accounts coa
        LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
        LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
        WHERE coa.account_type = 'expense'
        AND coa.account_code LIKE '5%'
    ");
    $stmt->execute([$as_of_date]);
    $expense_result = $stmt->fetch();
    $total_expense = $expense_result['total_expense'];

    // Calculate net income (revenues - expenses)
    $net_income = $total_revenue - $total_expense;

    // Calculate totals
    $total_assets = array_sum(array_column($asset_accounts, 'balance'));
    $total_liabilities = array_sum(array_column($liability_accounts, 'balance'));
    $total_equity = array_sum(array_column($equity_accounts, 'balance')) + $net_income;

    // Balance check (Assets should equal Liabilities + Equity)
    $balance_check = $total_assets - ($total_liabilities + $total_equity);

    // Student Fees Receivable balance is now calculated through journal entries only
    // Removed duplicate calculation from student_fees table to prevent double-counting

    // Group ALL asset accounts by their account codes for proper classification
    // Current assets: 1xxx accounts (except 12xx which are non-current)
    $current_assets = [];
    $non_current_assets = [];

    foreach ($asset_accounts as $account) {
        if (strpos($account['account_code'], '12') === 0) {
            // Non-current assets (12xx range)
            $non_current_assets[] = $account;
        } elseif (strpos($account['account_code'], '1') === 0) {
            // Current assets (1xxx range, excluding 12xx)
            $current_assets[] = $account;
        } else {
            // Other asset accounts - treat as current if not specified
            $current_assets[] = $account;
        }
    }

    // Add any missing cash accounts from cash_accounts table
    $stmt = $pdo->query("SELECT account_name, account_code, current_balance FROM cash_accounts");
    $cash_accounts = $stmt->fetchAll(PDO::FETCH_ASSOC);

    foreach ($cash_accounts as $cash_account) {
        $account_name = $cash_account['account_name'];
        $account_code = $cash_account['account_code'];
        $found = false;

        // Check if already in current assets
        foreach ($current_assets as $asset) {
            if ($asset['account_name'] == $account_name) {
                $found = true;
                break;
            }
        }

        if (!$found) {
            // Check if it exists in chart_of_accounts with journal entries
            $stmt = $pdo->prepare("
                SELECT
                    coa.id,
                    coa.account_code,
                    coa.account_name,
                    coa.account_type,
                    coa.normal_balance,
                    COALESCE(
                        SUM(
                            CASE
                                WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                                ELSE jel.credit_amount - jel.debit_amount
                            END
                        ), 0
                    ) as balance
                FROM chart_of_accounts coa
                LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
                LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
                WHERE coa.account_name = ? AND coa.account_type = 'asset'
                GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type, coa.normal_balance
            ");
            $stmt->execute([$as_of_date, $account_name]);
            $account = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($account && $account['balance'] != 0) {
                $current_assets[] = $account;
            } elseif ($cash_account['current_balance'] != 0) {
                // Add from cash_accounts table if no journal entries but has balance
                $current_assets[] = [
                    'account_code' => $account_code,
                    'account_name' => $account_name,
                    'balance' => $cash_account['current_balance']
                ];
            }
        }
    }

    // Sort assets by account code
    usort($current_assets, function($a, $b) {
        return strcmp($a['account_code'], $b['account_code']);
    });
    usort($non_current_assets, function($a, $b) {
        return strcmp($a['account_code'], $b['account_code']);
    });

    // Group ALL liability accounts by their account codes
    $current_liabilities = [];
    $non_current_liabilities = [];

    foreach ($liability_accounts as $account) {
        // Check for 'short' in account name first - overrides code-based grouping
        if (stripos($account['account_name'], 'short') !== false) {
            // Short-term loans and similar accounts are current liabilities
            $current_liabilities[] = $account;
        } elseif ((strpos($account['account_code'], '21') === 0 && $account['account_code'] !== '2100' && $account['account_code'] !== '2110') ||
            strpos($account['account_code'], '22') === 0 || strpos($account['account_code'], '23') === 0) {
            // Non-current liabilities (21xx except 2100 and 2110, 22xx, 23xx ranges)
            $non_current_liabilities[] = $account;
        } else {
            // Current liabilities (20xx range, 2100, 2110, and others)
            $current_liabilities[] = $account;
        }
    }

    // Sort liabilities by account code
    usort($current_liabilities, function($a, $b) {
        return strcmp($a['account_code'], $b['account_code']);
    });
    usort($non_current_liabilities, function($a, $b) {
        return strcmp($a['account_code'], $b['account_code']);
    });

    // Asset breakdown for chart
    $asset_breakdown = [
        ['category' => 'Current Assets', 'amount' => array_sum(array_column($current_assets, 'balance'))],
        ['category' => 'Non-Current Assets', 'amount' => array_sum(array_column($non_current_assets, 'balance'))]
    ];

    // Liability breakdown for chart
    $liability_breakdown = [
        ['category' => 'Current Liabilities', 'amount' => array_sum(array_column($current_liabilities, 'balance'))],
        ['category' => 'Non-Current Liabilities', 'amount' => array_sum(array_column($non_current_liabilities, 'balance'))],
        ['category' => 'Equity', 'amount' => $total_equity]
    ];



    // Monthly balance trends (last 6 months) - using journal entries
    $balance_trends = [];
    for ($i = 5; $i >= 0; $i--) {
        $month_end = date('Y-m-t', strtotime("-{$i} months"));
        $month_name = date('M Y', strtotime("-{$i} months"));

        // Calculate asset balances up to month end using journal entries
        $stmt = $pdo->prepare("
            SELECT
                coa.account_code,
                coa.account_name,
                COALESCE(
                    (SELECT SUM(
                        CASE
                            WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                            ELSE jel.credit_amount - jel.debit_amount
                        END
                    )
                    FROM journal_entries je
                    JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
                    WHERE jel.account_id = coa.id
                    AND je.transaction_date <= ?
                    AND je.status = 'posted'), 0
                ) as balance
            FROM chart_of_accounts coa
            WHERE coa.account_type = 'asset'
        ");
        $stmt->execute([$month_end]);
        $monthly_assets = $stmt->fetchAll();
        $total_assets_trend = array_sum(array_column($monthly_assets, 'balance'));

        // Calculate liability balances up to month end
        $stmt = $pdo->prepare("
            SELECT
                coa.account_code,
                coa.account_name,
                COALESCE(
                    (SELECT SUM(
                        CASE
                            WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                            ELSE jel.credit_amount - jel.debit_amount
                        END
                    )
                    FROM journal_entries je
                    JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
                    WHERE jel.account_id = coa.id
                    AND je.transaction_date <= ?
                    AND je.status = 'posted'), 0
                ) as balance
            FROM chart_of_accounts coa
            WHERE coa.account_type = 'liability'
        ");
        $stmt->execute([$month_end]);
        $monthly_liabilities = $stmt->fetchAll();
        $total_liabilities_trend = array_sum(array_column($monthly_liabilities, 'balance'));

        // Calculate equity balances up to month end
        $stmt = $pdo->prepare("
            SELECT
                COALESCE(
                    SUM(
                        CASE
                            WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                            ELSE jel.credit_amount - jel.debit_amount
                        END
                    ), 0
                ) as balance
            FROM chart_of_accounts coa
            LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
            LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
            WHERE coa.account_type = 'equity'
            AND coa.account_code LIKE '3%'
        ");
        $stmt->execute([$month_end]);
        $equity_result = $stmt->fetch();
        $total_equity_trend = $equity_result['balance'];

        // Calculate net income for the month (revenues - expenses)
        $stmt = $pdo->prepare("
            SELECT
                COALESCE(
                    SUM(
                        CASE
                            WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                            ELSE jel.credit_amount - jel.debit_amount
                        END
                    ), 0
                ) as total_revenue
            FROM chart_of_accounts coa
            LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
            LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
            WHERE coa.account_type = 'revenue'
            AND coa.account_code LIKE '4%'
        ");
        $stmt->execute([$month_end]);
        $revenue_result = $stmt->fetch();
        $total_revenue_trend = $revenue_result['total_revenue'];

        $stmt = $pdo->prepare("
            SELECT
                COALESCE(
                    SUM(
                        CASE
                            WHEN coa.normal_balance = 'debit' THEN jel.debit_amount - jel.credit_amount
                            ELSE jel.credit_amount - jel.debit_amount
                        END
                    ), 0
                ) as total_expense
            FROM chart_of_accounts coa
            LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
            LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.status = 'posted' AND je.transaction_date <= ?
            WHERE coa.account_type = 'expense'
            AND coa.account_code LIKE '5%'
        ");
        $stmt->execute([$month_end]);
        $expense_result = $stmt->fetch();
        $total_expense_trend = $expense_result['total_expense'];

        $net_income_trend = $total_revenue_trend - $total_expense_trend;
        $total_equity_trend += $net_income_trend;

        $balance_trends[] = [
            'month' => $month_name,
            'assets' => $total_assets_trend,
            'liabilities' => $total_liabilities_trend,
            'equity' => $total_equity_trend
        ];
    }

} catch (PDOException $e) {
    $error = "Error fetching balance sheet data: " . $e->getMessage();
}

// Fetch school settings for print functionality
$school_settings = $pdo->query("SELECT school_name, address, phone, email, logo_path FROM school_settings LIMIT 1")->fetch(PDO::FETCH_ASSOC);

include 'includes/accounting_header.php';
?>

<div class="balance-sheet-container">
    <div class="report-header">
        <h1>📊 Balance Sheet (Double-Entry System)</h1>
        <div class="report-filters">
            <form method="get" class="filter-form">
                <div class="filter-group">
                    <label for="report_type">Report Type:</label>
                    <select name="report_type" id="report_type" onchange="this.form.submit()">
                        <option value="current" <?php echo $report_type === 'current' ? 'selected' : ''; ?>>Current</option>
                        <option value="monthly" <?php echo $report_type === 'monthly' ? 'selected' : ''; ?>>Monthly</option>
                        <option value="yearly" <?php echo $report_type === 'yearly' ? 'selected' : ''; ?>>Yearly</option>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="as_of_date">As of Date:</label>
                    <input type="date" name="as_of_date" id="as_of_date" value="<?php echo $as_of_date; ?>" onchange="this.form.submit()">
                </div>

                <a href="export_balance_sheet.php?<?php echo http_build_query($_GET); ?>" class="btn btn-secondary">Export</a>
                <button type="button" onclick="printReport()" class="btn btn-primary">Print Report</button>
                <a href="year_end_summary.php" class="btn btn-info">Year End Summary</a>
            </form>
        </div>
    </div>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Balance Sheet Summary -->
    <div class="summary-cards">
        <div class="card assets-card">
            <div class="card-icon">🏢</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($total_assets, 2); ?></h3>
                <p>Total Assets</p>
                <small>What the school owns</small>
            </div>
        </div>

        <div class="card liabilities-card">
            <div class="card-icon">💳</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($total_liabilities, 2); ?></h3>
                <p>Total Liabilities</p>
                <small>What the school owes</small>
            </div>
        </div>

        <div class="card equity-card">
            <div class="card-icon">📈</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($total_equity, 2); ?></h3>
                <p>Total Equity</p>
                <small>Assets - Liabilities</small>
            </div>
        </div>

        <div class="card working-capital-card">
            <div class="card-icon">💰</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($total_assets - $total_liabilities, 2); ?></h3>
                <p>Working Capital</p>
                <small>Current assets - Current liabilities</small>
            </div>
        </div>

        <div class="card debt-ratio-card">
            <div class="card-icon">📊</div>
            <div class="card-content">
                <h3><?php echo $total_assets > 0 ? round(($total_liabilities / $total_assets) * 100, 1) : 0; ?>%</h3>
                <p>Debt Ratio</p>
                <small>Liabilities / Assets</small>
            </div>
        </div>

        <div class="card current-ratio-card">
            <div class="card-icon">⚖️</div>
            <div class="card-content">
                <h3><?php echo $total_liabilities > 0 ? round($total_assets / $total_liabilities, 2) : 0; ?></h3>
                <p>Current Ratio</p>
                <small>Assets / Liabilities</small>
            </div>
        </div>
    </div>

    <!-- Balance Sheet Statement -->
    <div class="report-section">
        <h2>Balance Sheet</h2>
        <p class="report-period">As of: <?php echo date('M d, Y', strtotime($as_of_date)); ?></p>

        <div class="balance-sheet-grid">
            <!-- Assets Section -->
            <div class="bs-section">
                <h3>Assets</h3>
                <div class="table-container">
                    <table class="bs-table">
                        <thead>
                            <tr>
                                <th>Account Code</th>
                                <th>Account Name</th>
                                <th>Amount (GHC)</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($current_assets)): ?>
                                <tr>
                                    <td colspan="3"><strong>Current Assets</strong></td>
                                </tr>
                                <?php foreach ($current_assets as $asset): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($asset['account_code']); ?></td>
                                        <td><?php echo htmlspecialchars($asset['account_name']); ?></td>
                                        <td class="positive"><?php echo number_format($asset['balance'], 2); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                                <tr class="subtotal-row">
                                    <td colspan="2"><strong>Subtotal Current Assets</strong></td>
                                    <td class="positive"><strong><?php echo number_format(array_sum(array_column($current_assets, 'balance')), 2); ?></strong></td>
                                </tr>
                            <?php endif; ?>

                            <?php if (!empty($non_current_assets)): ?>
                                <tr>
                                    <td colspan="3"><strong>Non-Current Assets</strong></td>
                                </tr>
                                <?php foreach ($non_current_assets as $asset): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($asset['account_code']); ?></td>
                                        <td><?php echo htmlspecialchars($asset['account_name']); ?></td>
                                        <td class="positive"><?php echo number_format($asset['balance'], 2); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                                <tr class="subtotal-row">
                                    <td colspan="2"><strong>Subtotal Non-Current Assets</strong></td>
                                    <td class="positive"><strong><?php echo number_format(array_sum(array_column($non_current_assets, 'balance')), 2); ?></strong></td>
                                </tr>
                            <?php endif; ?>

                            <tr class="total-row">
                                <td colspan="2"><strong>Total Assets</strong></td>
                                <td class="positive"><strong><?php echo number_format($total_assets, 2); ?></strong></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Liabilities & Equity Section -->
            <div class="bs-section">
                <h3>Liabilities & Equity</h3>
                <div class="table-container">
                    <table class="bs-table">
                        <thead>
                            <tr>
                                <th>Account Code</th>
                                <th>Account Name</th>
                                <th>Amount (GHC)</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($current_liabilities)): ?>
                                <tr>
                                    <td colspan="3"><strong>Current Liabilities</strong></td>
                                </tr>
                                <?php foreach ($current_liabilities as $liability): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($liability['account_code']); ?></td>
                                        <td><?php echo htmlspecialchars($liability['account_name']); ?></td>
                                        <td class="negative"><?php echo number_format($liability['balance'], 2); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                                <tr class="subtotal-row">
                                    <td colspan="2"><strong>Subtotal Current Liabilities</strong></td>
                                    <td class="negative"><strong><?php echo number_format(array_sum(array_column($current_liabilities, 'balance')), 2); ?></strong></td>
                                </tr>
                            <?php endif; ?>

                            <?php if (!empty($non_current_liabilities)): ?>
                                <tr>
                                    <td colspan="3"><strong>Non-Current Liabilities</strong></td>
                                </tr>
                                <?php foreach ($non_current_liabilities as $liability): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($liability['account_code']); ?></td>
                                        <td><?php echo htmlspecialchars($liability['account_name']); ?></td>
                                        <td class="negative"><?php echo number_format($liability['balance'], 2); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                                <tr class="subtotal-row">
                                    <td colspan="2"><strong>Subtotal Non-Current Liabilities</strong></td>
                                    <td class="negative"><strong><?php echo number_format(array_sum(array_column($non_current_liabilities, 'balance')), 2); ?></strong></td>
                                </tr>
                            <?php endif; ?>

                            <tr class="total-row">
                                <td colspan="2"><strong>Total Liabilities</strong></td>
                                <td class="negative"><strong><?php echo number_format($total_liabilities, 2); ?></strong></td>
                            </tr>

                            <?php if (!empty($equity_accounts)): ?>
                                <tr>
                                    <td colspan="3"><strong>Equity</strong></td>
                                </tr>
                                <?php foreach ($equity_accounts as $equity): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($equity['account_code']); ?></td>
                                        <td><?php echo htmlspecialchars($equity['account_name']); ?></td>
                                        <td class="<?php echo $equity['balance'] >= 0 ? 'positive' : 'negative'; ?>"><?php echo number_format($equity['balance'], 2); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>

                            <tr class="total-row">
                                <td colspan="2"><strong>Total Equity</strong></td>
                                <td class="<?php echo $total_equity >= 0 ? 'positive' : 'negative'; ?>">
                                    <strong><?php echo number_format($total_equity, 2); ?></strong>
                                </td>
                            </tr>
                            <tr class="total-row final-total">
                                <td colspan="2"><strong>Total Liabilities & Equity</strong></td>
                                <td><strong><?php echo number_format($total_liabilities + $total_equity, 2); ?></strong></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Balance Check -->
        <div class="balance-check">
            <div class="table-container">
                <table class="bs-table">
                    <tr class="balance-check-row">
                        <td><strong>Balance Check:</strong></td>
                        <td class="<?php echo abs($balance_check) < 0.01 ? 'balanced' : 'unbalanced'; ?>">
                            <?php echo abs($balance_check) < 0.01 ? '✅ Balanced' : '❌ Unbalanced (Difference: GHC ' . number_format($balance_check, 2) . ')'; ?>
                        </td>
                    </tr>
                </table>
            </div>
        </div>
    </div>

    <!-- Journal Entry Summary -->
    <div class="report-section">
        <h2>📝 Recent Journal Entries</h2>
        <div class="table-container">
            <table class="bs-table">
                <thead>
                    <tr>
                        <th>Entry #</th>
                        <th>Date</th>
                        <th>Description</th>
                        <th>Source</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $stmt = $pdo->prepare("
                        SELECT je.entry_number, je.transaction_date, je.description, je.source_type, je.status
                        FROM journal_entries je
                        WHERE je.status = 'posted'
                        ORDER BY je.id DESC
                        LIMIT 10
                    ");
                    $stmt->execute();
                    $recent_entries = $stmt->fetchAll();

                    if (empty($recent_entries)): ?>
                        <tr>
                            <td colspan="5" style="text-align: center; color: #7f8c8d;">No journal entries found</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($recent_entries as $entry): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($entry['entry_number']); ?></td>
                                <td><?php echo date('M d, Y', strtotime($entry['transaction_date'])); ?></td>
                                <td><?php echo htmlspecialchars(substr($entry['description'], 0, 50)) . (strlen($entry['description']) > 50 ? '...' : ''); ?></td>
                                <td><?php echo ucfirst(htmlspecialchars($entry['source_type'])); ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo $entry['status']; ?>">
                                        <?php echo ucfirst($entry['status']); ?>
                                    </span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Charts Section -->
    <div class="charts-section">
        <div class="chart-container">
            <h3>Assets Breakdown</h3>
            <canvas id="assetsChart" width="400" height="200"></canvas>
        </div>

        <div class="chart-container">
            <h3>Liabilities & Equity</h3>
            <canvas id="liabilitiesChart" width="400" height="200"></canvas>
        </div>
    </div>

    <!-- Balance Trends Chart -->
    <div class="report-section">
        <h2>Balance Sheet Trends (Last 6 Months)</h2>
        <div class="chart-container">
            <canvas id="balanceTrendChart" width="400" height="200"></canvas>
        </div>
    </div>

    <!-- Financial Health Analysis -->
    <div class="report-section">
        <h2>💡 Financial Health Analysis</h2>
        <div class="analysis-grid">
            <div class="analysis-card">
                <h4>Financial Position</h4>
                <p class="<?php echo $total_equity >= 0 ? 'positive' : 'negative'; ?>">
                    <?php echo $total_equity >= 0 ? 'Solvent' : 'Insolvent'; ?>
                </p>
                <small>
                    <?php
                    if ($total_equity >= 0) {
                        echo 'Assets exceed liabilities - healthy financial position';
                    } else {
                        echo 'Liabilities exceed assets - requires immediate attention';
                    }
                    ?>
                </small>
            </div>

            <div class="analysis-card">
                <h4>Debt Ratio</h4>
                <p class="<?php echo $total_assets > 0 && ($total_liabilities / $total_assets) * 100 <= 50 ? 'good' : 'warning'; ?>">
                    <?php echo $total_assets > 0 ? round(($total_liabilities / $total_assets) * 100, 1) : 0; ?>%
                </p>
                <small>
                    <?php
                    if ($total_assets > 0) {
                        $debt_ratio = ($total_liabilities / $total_assets) * 100;
                        if ($debt_ratio <= 30) echo 'Excellent - Low debt burden';
                        elseif ($debt_ratio <= 50) echo 'Good - Manageable debt level';
                        elseif ($debt_ratio <= 70) echo 'Fair - Moderate debt level';
                        else echo 'High - Consider debt reduction strategies';
                    } else {
                        echo 'Cannot calculate - No assets';
                    }
                    ?>
                </small>
            </div>

            <div class="analysis-card">
                <h4>Current Ratio</h4>
                <p class="<?php echo $total_liabilities > 0 && $total_assets / $total_liabilities >= 1.5 ? 'excellent' : ($total_liabilities > 0 && $total_assets / $total_liabilities >= 1 ? 'good' : 'poor'); ?>">
                    <?php echo $total_liabilities > 0 ? round($total_assets / $total_liabilities, 2) : 'N/A'; ?>
                </p>
                <small>
                    <?php
                    if ($total_liabilities > 0) {
                        $current_ratio = $total_assets / $total_liabilities;
                        if ($current_ratio >= 2) echo 'Excellent - Strong liquidity position';
                        elseif ($current_ratio >= 1.5) echo 'Good - Healthy liquidity';
                        elseif ($current_ratio >= 1) echo 'Fair - Adequate liquidity';
                        else echo 'Poor - Liquidity concerns';
                    } else {
                        echo 'Cannot calculate - No liabilities';
                    }
                    ?>
                </small>
            </div>

            <div class="analysis-card">
                <h4>Working Capital</h4>
                <p class="<?php echo $total_assets - $total_liabilities >= 0 ? 'positive' : 'negative'; ?>">
                    GHC <?php echo number_format($total_assets - $total_liabilities, 2); ?>
                </p>
                <small>
                    <?php
                    $working_capital = $total_assets - $total_liabilities;
                    if ($working_capital > 0) {
                        echo 'Positive working capital - can meet short-term obligations';
                    } else {
                        echo 'Negative working capital - may face liquidity issues';
                    }
                    ?>
                </small>
            </div>
        </div>
    </div>
</div>

<style>
.balance-sheet-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.report-header {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-header h1 {
    color: #2c3e50;
    margin: 0 0 20px 0;
}

.filter-form {
    display: flex;
    gap: 20px;
    align-items: end;
    flex-wrap: wrap;
}

.filter-group {
    display: flex;
    flex-direction: column;
    min-width: 150px;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group select,
.filter-group input {
    padding: 8px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.card {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
    transition: transform 0.2s;
}

.card:hover {
    transform: translateY(-5px);
}

.card-icon {
    font-size: 2.5em;
    margin-bottom: 15px;
}

.card h3 {
    color: #2c3e50;
    font-size: 1.8em;
    margin-bottom: 5px;
}

.card p {
    color: #7f8c8d;
    font-size: 1.1em;
    margin: 0;
}

.card small {
    color: #95a5a6;
    font-size: 0.9em;
}

.assets-card .card-icon { color: #27ae60; }
.liabilities-card .card-icon { color: #e74c3c; }
.equity-card .card-icon { color: #3498db; }
.working-capital-card .card-icon { color: #9b59b6; }
.debt-ratio-card .card-icon,
.current-ratio-card .card-icon { color: #f39c12; }

.report-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.report-section h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    border-bottom: 2px solid #ecf0f1;
    padding-bottom: 10px;
}

.report-period {
    color: #7f8c8d;
    font-style: italic;
    margin-bottom: 20px;
}

.balance-sheet-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 30px;
    margin-bottom: 30px;
}

.bs-section h3 {
    color: #2c3e50;
    margin-bottom: 15px;
    font-size: 1.2em;
}

.table-container {
    overflow-x: auto;
}

.bs-table {
    width: 100%;
    border-collapse: collapse;
    margin-bottom: 20px;
}

.bs-table th,
.bs-table td {
    padding: 12px;
    text-align: left;
    border-bottom: 1px solid #ecf0f1;
}

.bs-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.positive {
    color: #27ae60;
    font-weight: 600;
}

.negative {
    color: #e74c3c;
    font-weight: 600;
}

.total-row {
    background: #f8f9fa;
    font-weight: 600;
}

.subtotal-row {
    background: #f1f3f4;
    font-weight: 600;
}

.final-total {
    background: #ecf0f1;
    border-top: 2px solid #bdc3c7;
}

.balance-check {
    margin-top: 20px;
    padding: 15px;
    background: #f8f9fa;
    border-radius: 5px;
}

.balance-check-row td {
    padding: 10px;
    border: none;
}

.balanced {
    color: #27ae60;
    font-weight: 600;
}

.unbalanced {
    color: #e74c3c;
    font-weight: 600;
}

.status-badge {
    display: inline-block;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 0.8em;
    font-weight: 600;
    text-transform: uppercase;
    border: 1px solid;
}

.status-posted {
    background: #d4edda;
    color: #155724;
    border-color: #c3e6cb;
}

.status-draft {
    background: #fff3cd;
    color: #856404;
    border-color: #ffeaa7;
}

.charts-section {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 30px;
    margin-bottom: 30px;
}

.chart-container {
    background: white;
    padding: 20px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.chart-container h3 {
    color: #2c3e50;
    margin-bottom: 15px;
    text-align: center;
}

.analysis-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
}

.analysis-card {
    background: white;
    padding: 20px;
    border-radius: 8px;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    border-left: 4px solid #3498db;
}

.analysis-card h4 {
    color: #2c3e50;
    margin: 0 0 10px 0;
    font-size: 1.1em;
}

.analysis-card p {
    font-size: 1.2em;
    font-weight: 600;
    margin: 5px 0;
}

.analysis-card small {
    color: #7f8c8d;
    font-size: 0.9em;
    line-height: 1.4;
}

.good { color: #27ae60; }
.warning { color: #f39c12; }
.excellent { color: #8e44ad; }
.poor { color: #e74c3c; }

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

.btn {
    display: inline-block;
    padding: 10px 20px;
    background: #3498db;
    color: white;
    text-decoration: none;
    border-radius: 5px;
    transition: background 0.3s;
}

.btn:hover {
    background: #2980b9;
}

.btn-secondary {
    background: #95a5a6;
}

.btn-secondary:hover {
    background: #7f8c8d;
}

@media (max-width: 768px) {
    .balance-sheet-grid,
    .charts-section {
        grid-template-columns: 1fr;
    }

    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }

    .summary-cards {
        grid-template-columns: 1fr;
    }
}
</style>

<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<script>
// Pass PHP data to JavaScript
const phpData = {
    current_assets: <?php echo json_encode($current_assets); ?>,
    non_current_assets: <?php echo json_encode($non_current_assets); ?>,
    current_liabilities: <?php echo json_encode($current_liabilities); ?>,
    non_current_liabilities: <?php echo json_encode($non_current_liabilities); ?>,
    equity_accounts: <?php echo json_encode($equity_accounts); ?>,
    total_assets: <?php echo json_encode($total_assets); ?>,
    total_liabilities: <?php echo json_encode($total_liabilities); ?>,
    total_equity: <?php echo json_encode($total_equity); ?>,
    balance_check: <?php echo json_encode($balance_check); ?>,
    as_of_date: <?php echo json_encode($as_of_date); ?>,
    school_settings: <?php echo json_encode($school_settings); ?>
};

// Assets Chart
const assetsCtx = document.getElementById('assetsChart').getContext('2d');
const assetsChart = new Chart(assetsCtx, {
    type: 'pie',
    data: {
        labels: <?php echo json_encode(array_column($asset_breakdown, 'category')); ?>,
        datasets: [{
            data: <?php echo json_encode(array_column($asset_breakdown, 'amount')); ?>,
            backgroundColor: ['#27ae60', '#3498db'],
            borderWidth: 1
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: {
                position: 'top',
            },
            tooltip: {
                callbacks: {
                    label: function(context) {
                        return context.label + ': GHC ' + context.parsed.toLocaleString();
                    }
                }
            }
        }
    }
});

// Liabilities Chart
const liabilitiesCtx = document.getElementById('liabilitiesChart').getContext('2d');
const liabilitiesChart = new Chart(liabilitiesCtx, {
    type: 'pie',
    data: {
        labels: <?php echo json_encode(array_column($liability_breakdown, 'category')); ?>,
        datasets: [{
            data: <?php echo json_encode(array_column($liability_breakdown, 'amount')); ?>,
            backgroundColor: ['#e74c3c', '#f39c12'],
            borderWidth: 1
        }]
    },
    options: {
        responsive: true,
        plugins: {
            legend: {
                position: 'top',
            },
            tooltip: {
                callbacks: {
                    label: function(context) {
                        return context.label + ': GHC ' + context.parsed.toLocaleString();
                    }
                }
            }
        }
    }
});

// Balance Trends Chart
const balanceTrendCtx = document.getElementById('balanceTrendChart').getContext('2d');
const balanceTrendChart = new Chart(balanceTrendCtx, {
    type: 'line',
    data: {
        labels: <?php echo json_encode(array_column($balance_trends, 'month')); ?>,
        datasets: [
            {
                label: 'Assets',
                data: <?php echo json_encode(array_column($balance_trends, 'assets')); ?>,
                borderColor: '#27ae60',
                backgroundColor: 'rgba(39, 174, 96, 0.1)',
                tension: 0.4
            },
            {
                label: 'Liabilities',
                data: <?php echo json_encode(array_column($balance_trends, 'liabilities')); ?>,
                borderColor: '#e74c3c',
                backgroundColor: 'rgba(231, 76, 60, 0.1)',
                tension: 0.4
            },
            {
                label: 'Equity',
                data: <?php echo json_encode(array_column($balance_trends, 'equity')); ?>,
                borderColor: '#3498db',
                backgroundColor: 'rgba(52, 152, 219, 0.1)',
                tension: 0.4
            }
        ]
    },
    options: {
        responsive: true,
        plugins: {
            legend: {
                position: 'top',
            },
            tooltip: {
                callbacks: {
                    label: function(context) {
                        return context.dataset.label + ': GHC ' + context.parsed.y.toLocaleString();
                    }
                }
            }
        },
        scales: {
            y: {
                beginAtZero: true,
                ticks: {
                    callback: function(value) {
                        return 'GHC ' + value.toLocaleString();
                    }
                }
            }
        }
    }
});

// Print Report Function
function printReport() {
    // Create a new window for printing
    var printWindow = window.open('', '_blank');

    // Get data from phpData
    var currentAssets = phpData.current_assets || [];
    var nonCurrentAssets = phpData.non_current_assets || [];
    var currentLiabilities = phpData.current_liabilities || [];
    var nonCurrentLiabilities = phpData.non_current_liabilities || [];
    var equityAccounts = phpData.equity_accounts || [];
    var totalAssets = phpData.total_assets || 0;
    var totalLiabilities = phpData.total_liabilities || 0;
    var totalEquity = phpData.total_equity || 0;
    var balanceCheck = phpData.balance_check || 0;
    var schoolSettings = phpData.school_settings || {};

    // Build the print content using JavaScript
    var printContent = `
        <!DOCTYPE html>
        <html>
        <head>
            <title>Balance Sheet Report</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    margin: 20px;
                    line-height: 1.4;
                }
                .header {
                    text-align: center;
                    border-bottom: 2px solid #333;
                    padding-bottom: 20px;
                    margin-bottom: 30px;
                }
                .school-info {
                    margin-bottom: 20px;
                }
                .report-title {
                    font-size: 24px;
                    font-weight: bold;
                    margin: 20px 0;
                    color: #2c3e50;
                }
                .summary-section {
                    display: flex;
                    justify-content: space-around;
                    margin-bottom: 30px;
                    padding: 20px;
                    background: #f8f9fa;
                    border-radius: 8px;
                }
                .summary-item {
                    text-align: center;
                    flex: 1;
                }
                .summary-item h3 {
                    font-size: 28px;
                    margin: 10px 0;
                    color: #2c3e50;
                }
                .summary-item p {
                    color: #7f8c8d;
                    margin: 5px 0;
                    font-size: 14px;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin-top: 20px;
                    font-size: 12px;
                }
                th, td {
                    border: 1px solid #ddd;
                    padding: 8px 12px;
                    text-align: left;
                }
                th {
                    background-color: #f8f9fa;
                    font-weight: bold;
                    color: #2c3e50;
                }
                .positive {
                    color: #27ae60;
                    font-weight: bold;
                }
                .negative {
                    color: #e74c3c;
                    font-weight: bold;
                }
                .total-row {
                    background-color: #e9ecef;
                    font-weight: bold;
                }
                .subtotal-row {
                    background-color: #f1f3f4;
                    font-weight: bold;
                }
                .final-total {
                    background-color: #ecf0f1;
                    border-top: 2px solid #bdc3c7;
                }
                .balance-check {
                    margin-top: 20px;
                    padding: 15px;
                    background: #f8f9fa;
                    border-radius: 5px;
                    text-align: center;
                }
                .balanced {
                    color: #27ae60;
                    font-weight: bold;
                }
                .unbalanced {
                    color: #e74c3c;
                    font-weight: bold;
                }
                .footer {
                    margin-top: 40px;
                    text-align: center;
                    font-size: 12px;
                    color: #6c757d;
                    border-top: 1px solid #dee2e6;
                    padding-top: 20px;
                }
                @media print {
                    body { margin: 0; }
                    .no-print { display: none; }
                }
            </style>
        </head>
        <body>
            <div class="header">
                <div class="school-info">
                    <h1>${schoolSettings.school_name || 'School Name'}</h1>
                    <p>${schoolSettings.address || 'School Address'}</p>
                    <p>Phone: ${schoolSettings.phone || 'School Phone'} | Email: ${schoolSettings.email || 'School Email'}</p>
                </div>
                <h2 class="report-title">Balance Sheet Report</h2>
                <p>As of: ${phpData.as_of_date ? new Date(phpData.as_of_date).toLocaleDateString('en-US', { year: 'numeric', month: 'short', day: 'numeric' }) : 'N/A'}</p>
                <p>Generated on: ${new Date().toLocaleDateString()} ${new Date().toLocaleTimeString()}</p>
            </div>

            <div class="summary-section">
                <div class="summary-item">
                    <h3>GHC ${totalAssets.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</h3>
                    <p>Total Assets</p>
                </div>
                <div class="summary-item">
                    <h3>GHC ${totalLiabilities.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</h3>
                    <p>Total Liabilities</p>
                </div>
                <div class="summary-item">
                    <h3>GHC ${totalEquity.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</h3>
                    <p>Total Equity</p>
                </div>
                <div class="summary-item">
                    <h3>${totalAssets > 0 ? ((totalLiabilities / totalAssets) * 100).toFixed(1) : 0}%</h3>
                    <p>Debt Ratio</p>
                </div>
            </div>

            <table>
                <thead>
                    <tr>
                        <th colspan="3" style="text-align: center; background: #f8f9fa;">ASSETS</th>
                    </tr>
                    <tr>
                        <th>Account Code</th>
                        <th>Account Name</th>
                        <th>Amount (GHC)</th>
                    </tr>
                </thead>
                <tbody>`;

    if (currentAssets.length > 0) {
        printContent += `
                    <tr>
                        <td colspan="3"><strong>Current Assets</strong></td>
                    </tr>`;
        var currentAssetsTotal = 0;
        currentAssets.forEach(function(asset) {
            printContent += `
                    <tr>
                        <td>${asset.account_code || ''}</td>
                        <td>${asset.account_name || ''}</td>
                        <td class="positive">${parseFloat(asset.balance || 0).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td>
                    </tr>`;
            currentAssetsTotal += parseFloat(asset.balance || 0);
        });
        printContent += `
                    <tr class="subtotal-row">
                        <td colspan="2"><strong>Subtotal Current Assets</strong></td>
                        <td class="positive"><strong>${currentAssetsTotal.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</strong></td>
                    </tr>`;
    }

    if (nonCurrentAssets.length > 0) {
        printContent += `
                    <tr>
                        <td colspan="3"><strong>Non-Current Assets</strong></td>
                    </tr>`;
        var nonCurrentAssetsTotal = 0;
        nonCurrentAssets.forEach(function(asset) {
            printContent += `
                    <tr>
                        <td>${asset.account_code || ''}</td>
                        <td>${asset.account_name || ''}</td>
                        <td class="positive">${parseFloat(asset.balance || 0).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td>
                    </tr>`;
            nonCurrentAssetsTotal += parseFloat(asset.balance || 0);
        });
        printContent += `
                    <tr class="subtotal-row">
                        <td colspan="2"><strong>Subtotal Non-Current Assets</strong></td>
                        <td class="positive"><strong>${nonCurrentAssetsTotal.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</strong></td>
                    </tr>`;
    }

    printContent += `
                    <tr class="total-row">
                        <td colspan="2"><strong>Total Assets</strong></td>
                        <td class="positive"><strong>${totalAssets.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</strong></td>
                    </tr>
                </tbody>
            </table>

            <table style="margin-top: 30px;">
                <thead>
                    <tr>
                        <th colspan="3" style="text-align: center; background: #f8f9fa;">LIABILITIES & EQUITY</th>
                    </tr>
                    <tr>
                        <th>Account Code</th>
                        <th>Account Name</th>
                        <th>Amount (GHC)</th>
                    </tr>
                </thead>
                <tbody>`;

    if (currentLiabilities.length > 0) {
        printContent += `
                    <tr>
                        <td colspan="3"><strong>Current Liabilities</strong></td>
                    </tr>`;
        var currentLiabilitiesTotal = 0;
        currentLiabilities.forEach(function(liability) {
            printContent += `
                    <tr>
                        <td>${liability.account_code || ''}</td>
                        <td>${liability.account_name || ''}</td>
                        <td class="negative">${parseFloat(liability.balance || 0).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td>
                    </tr>`;
            currentLiabilitiesTotal += parseFloat(liability.balance || 0);
        });
        printContent += `
                    <tr class="subtotal-row">
                        <td colspan="2"><strong>Subtotal Current Liabilities</strong></td>
                        <td class="negative"><strong>${currentLiabilitiesTotal.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</strong></td>
                    </tr>`;
    }

    if (nonCurrentLiabilities.length > 0) {
        printContent += `
                    <tr>
                        <td colspan="3"><strong>Non-Current Liabilities</strong></td>
                    </tr>`;
        var nonCurrentLiabilitiesTotal = 0;
        nonCurrentLiabilities.forEach(function(liability) {
            printContent += `
                    <tr>
                        <td>${liability.account_code || ''}</td>
                        <td>${liability.account_name || ''}</td>
                        <td class="negative">${parseFloat(liability.balance || 0).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td>
                    </tr>`;
            nonCurrentLiabilitiesTotal += parseFloat(liability.balance || 0);
        });
        printContent += `
                    <tr class="subtotal-row">
                        <td colspan="2"><strong>Subtotal Non-Current Liabilities</strong></td>
                        <td class="negative"><strong>${nonCurrentLiabilitiesTotal.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</strong></td>
                    </tr>`;
    }

    printContent += `
                    <tr class="total-row">
                        <td colspan="2"><strong>Total Liabilities</strong></td>
                        <td class="negative"><strong>${totalLiabilities.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</strong></td>
                    </tr>`;

    if (equityAccounts.length > 0) {
        printContent += `
                    <tr>
                        <td colspan="3"><strong>Equity</strong></td>
                    </tr>`;
        equityAccounts.forEach(function(equity) {
            printContent += `
                    <tr>
                        <td>${equity.account_code || ''}</td>
                        <td>${equity.account_name || ''}</td>
                        <td class="${parseFloat(equity.balance || 0) >= 0 ? 'positive' : 'negative'}">${parseFloat(equity.balance || 0).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</td>
                    </tr>`;
        });
    }

    printContent += `
                    <tr class="total-row">
                        <td colspan="2"><strong>Total Equity</strong></td>
                        <td class="${totalEquity >= 0 ? 'positive' : 'negative'}">
                            <strong>${totalEquity.toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</strong>
                        </td>
                    </tr>
                    <tr class="total-row final-total">
                        <td colspan="2"><strong>Total Liabilities & Equity</strong></td>
                        <td><strong>${(totalLiabilities + totalEquity).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2})}</strong></td>
                    </tr>
                </tbody>
            </table>

            <div class="balance-check">
                <strong>Balance Check:</strong>
                <span class="${Math.abs(balanceCheck) < 0.01 ? 'balanced' : 'unbalanced'}">
                    ${Math.abs(balanceCheck) < 0.01 ? '✅ Balanced' : '❌ Unbalanced (Difference: GHC ' + Math.abs(balanceCheck).toLocaleString(undefined, {minimumFractionDigits: 2, maximumFractionDigits: 2}) + ')'}
                </span>
            </div>

            <div class="footer">
                <p>This report was generated by the School Management System</p>
                <p>Report Date: ${new Date().toLocaleDateString()}</p>
            </div>
        </body>
        </html>`;

    // Write content to the new window
    printWindow.document.write(printContent);
    printWindow.document.close();

    // Wait for content to load then print
    printWindow.onload = function() {
        printWindow.print();
        printWindow.close();
    };
}
</script>

<?php include 'includes/accounting_footer.php'; ?>
