<?php
session_start();

// Check permissions - only admins and accounts users can approve purchase orders
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['approve_po'])) {
        // Approve purchase order
        $po_id = intval($_POST['po_id']);

        try {
            $stmt = $pdo->prepare("
                UPDATE purchase_orders
                SET status = 'approved', updated_at = NOW()
                WHERE id = ? AND status = 'pending'
            ");
            $stmt->execute([$po_id]);

            if ($stmt->rowCount() > 0) {
                // Log activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'po_approved',
                    "Approved purchase order ID $po_id"
                );

                $message = "Purchase order approved successfully!";
            } else {
                $error = "Purchase order not found or not ready for approval.";
            }
        } catch (PDOException $e) {
            $error = "Error approving purchase order: " . $e->getMessage();
        }
    } elseif (isset($_POST['reject_po'])) {
        // Reject purchase order (send back to pending)
        $po_id = intval($_POST['po_id']);

        try {
            $stmt = $pdo->prepare("
                UPDATE purchase_orders
                SET status = 'pending', updated_at = NOW()
                WHERE id = ? AND status = 'pending'
            ");
            $stmt->execute([$po_id]);

            if ($stmt->rowCount() > 0) {
                // Log activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'po_rejected',
                    "Rejected purchase order ID $po_id - sent back to pending"
                );

                $message = "Purchase order rejected and sent back to pending status.";
            } else {
                $error = "Purchase order not found or not ready for approval.";
            }
        } catch (PDOException $e) {
            $error = "Error rejecting purchase order: " . $e->getMessage();
        }
    }
}

// Get purchase orders ready for approval (pending status)
try {
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';

    $query = "
        SELECT po.*, s.name AS supplier_name,
               u.username AS created_by_name,
               (
                   SELECT COUNT(*) FROM purchase_order_items poi WHERE poi.purchase_order_id = po.id
               ) as item_count,
               (
                   SELECT COALESCE(SUM(pop.amount), 0) FROM purchase_order_payments pop WHERE pop.purchase_order_id = po.id
               ) as total_paid
        FROM purchase_orders po
        JOIN suppliers s ON po.supplier_id = s.id
        LEFT JOIN users u ON po.created_by = u.id
        WHERE po.status = 'pending'
    ";

    $params = [];

    if (!empty($search)) {
        $query .= " AND (po.po_number LIKE ? OR s.name LIKE ?)";
        $params[] = "%$search%";
        $params[] = "%$search%";
    }

    $query .= " ORDER BY po.created_at DESC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $purchase_orders = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading purchase orders: " . $e->getMessage();
    $purchase_orders = [];
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .approve-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .approve-header {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .approve-content {
        display: grid;
        grid-template-columns: 1fr;
        gap: 30px;
    }

    .approve-section {
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .section-title {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
        font-size: 1.5em;
    }

    .filters-section {
        display: flex;
        gap: 20px;
        margin-bottom: 20px;
        align-items: center;
    }

    .filter-group {
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .filter-group label {
        font-weight: 600;
        color: #495057;
    }

    .filter-group input {
        padding: 8px 12px;
        border: 1px solid #ced4da;
        border-radius: 5px;
        font-size: 14px;
    }

    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        margin-right: 10px;
        transition: all 0.3s ease;
    }

    .btn-primary {
        background: #007bff;
        color: white;
    }

    .btn-primary:hover {
        background: #0056b3;
        transform: translateY(-2px);
    }

    .btn-success {
        background: #28a745;
        color: white;
    }

    .btn-success:hover {
        background: #1e7e34;
        transform: translateY(-2px);
    }

    .btn-danger {
        background: #dc3545;
        color: white;
    }

    .btn-danger:hover {
        background: #c82333;
        transform: translateY(-2px);
    }

    .btn-secondary {
        background: #6c757d;
        color: white;
    }

    .btn-secondary:hover {
        background: #545b62;
        transform: translateY(-2px);
    }

    .message {
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }

    .success-message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .po-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .po-table th,
    .po-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .po-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #495057;
    }

    .po-number {
        font-weight: bold;
        color: #007bff;
    }

    .supplier-name {
        font-weight: 600;
        color: #28a745;
    }

    .status-badge {
        display: inline-block;
        padding: 4px 12px;
        border-radius: 20px;
        font-size: 0.85em;
        font-weight: 600;
        text-transform: uppercase;
        background: #d4edda;
        color: #155724;
    }

    .amount {
        font-weight: 600;
        color: #007bff;
    }

    .actions-column {
        width: 250px;
    }

    .btn-approve {
        background: #28a745;
        color: white;
        border: none;
        padding: 6px 12px;
        border-radius: 4px;
        cursor: pointer;
        font-size: 12px;
        margin-right: 5px;
    }

    .btn-approve:hover {
        background: #218838;
    }

    .btn-reject {
        background: #dc3545;
        color: white;
        border: none;
        padding: 6px 12px;
        border-radius: 4px;
        cursor: pointer;
        font-size: 12px;
    }

    .btn-reject:hover {
        background: #c82333;
    }

    .no-pos {
        text-align: center;
        padding: 40px;
        color: #6c757d;
    }

    .no-pos h3 {
        margin-bottom: 10px;
        color: #495057;
    }

    @media (max-width: 768px) {
        .filters-section {
            flex-direction: column;
            align-items: stretch;
        }

        .po-table {
            font-size: 0.9em;
        }

        .actions-column {
            width: 200px;
        }
    }
</style>

<div class="approve-container">
    <div class="approve-header">
        <h1>✅ Approve Purchase Orders</h1>
        <p>Review and approve purchase orders that are pending for approval</p>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="approve-content">
        <div class="approve-section">
            <div class="section-title">📋 Purchase Orders Ready for Approval</div>

            <div class="filters-section">
                <div class="filter-group">
                    <label for="search">Search:</label>
                    <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($search ?? ''); ?>" placeholder="PO number or supplier...">
                </div>
                <button class="btn btn-primary" onclick="applyFilters()">🔍 Filter</button>
                <a href="purchase_orders.php" class="btn btn-secondary">← Back to All POs</a>
            </div>

            <?php if (empty($purchase_orders)): ?>
                <div class="no-pos">
                    <h3>No Purchase Orders Ready for Approval</h3>
                    <p>All purchase orders have been processed. Check back later for new deliveries.</p>
                </div>
            <?php else: ?>
                <table class="po-table">
                    <thead>
                        <tr>
                            <th>PO Number</th>
                            <th>Supplier</th>
                            <th>Items</th>
                            <th>Total Amount</th>
                            <th>Expected Delivery</th>
                            <th>Status</th>
                            <th>Created</th>
                            <th class="actions-column">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($purchase_orders as $po): ?>
                            <tr>
                                <td>
                                    <div class="po-number"><?php echo htmlspecialchars($po['po_number']); ?></div>
                                </td>
                                <td>
                                    <div class="supplier-name"><?php echo htmlspecialchars($po['supplier_name']); ?></div>
                                </td>
                                <td><?php echo $po['item_count']; ?> items</td>
                                <td class="amount">GHC <?php echo number_format($po['total_amount'], 2); ?></td>
                                <td><?php echo date('M j, Y', strtotime($po['expected_delivery_date'])); ?></td>
                                <td>
                                    <span class="status-badge">Pending</span>
                                </td>
                                <td><?php echo date('M j, Y', strtotime($po['created_at'])); ?></td>
                                <td class="actions-column">
                                    <button class="btn-approve" onclick="approvePO(<?php echo $po['id']; ?>, '<?php echo htmlspecialchars($po['po_number']); ?>')">✅ Approve</button>
                                    <button class="btn-reject" onclick="rejectPO(<?php echo $po['id']; ?>, '<?php echo htmlspecialchars($po['po_number']); ?>')">❌ Reject</button>
                                    <a href="print_po.php?po_id=<?php echo $po['id']; ?>" target="_blank" class="btn btn-secondary btn-sm" style="margin-left: 5px;">👁️ View</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Approval Modal -->
<div id="approvalModal" class="modal">
    <div class="modal-content approval-modal">
        <div class="modal-header">
            <div class="modal-title-section">
                <div class="modal-icon">✅</div>
                <div>
                    <h2 class="modal-title">Confirm Approval</h2>
                    <p class="modal-subtitle">Are you sure you want to approve this purchase order?</p>
                </div>
            </div>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>

        <div class="modal-body">
            <p id="approvalText"></p>
        </div>

        <form method="post" class="approval-form">
            <input type="hidden" name="po_id" id="approval_po_id">

            <div class="form-actions">
                <button type="button" class="btn btn-secondary" onclick="closeModal()">
                    <i class="fas fa-times"></i>
                    Cancel
                </button>
                <button type="submit" name="approve_po" class="btn btn-success">
                    <i class="fas fa-check"></i>
                    Yes, Approve
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Rejection Modal -->
<div id="rejectionModal" class="modal">
    <div class="modal-content rejection-modal">
        <div class="modal-header">
            <div class="modal-title-section">
                <div class="modal-icon">❌</div>
                <div>
                    <h2 class="modal-title">Confirm Rejection</h2>
                    <p class="modal-subtitle">Are you sure you want to reject this purchase order?</p>
                </div>
            </div>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>

        <div class="modal-body">
            <p id="rejectionText"></p>
            <p><strong>Note:</strong> Rejected purchase orders will be sent back to pending status.</p>
        </div>

        <form method="post" class="rejection-form">
            <input type="hidden" name="po_id" id="rejection_po_id">

            <div class="form-actions">
                <button type="button" class="btn btn-secondary" onclick="closeModal()">
                    <i class="fas fa-times"></i>
                    Cancel
                </button>
                <button type="submit" name="reject_po" class="btn btn-danger">
                    <i class="fas fa-times"></i>
                    Yes, Reject
                </button>
            </div>
        </form>
    </div>
</div>

<style>
/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
}

.modal-content {
    background-color: #fefefe;
    margin: 10% auto;
    border: 1px solid #888;
    border-radius: 10px;
    max-width: 500px;
    width: 90%;
}

.modal-header {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
    padding: 20px;
    border-radius: 10px 10px 0 0;
}

.rejection-modal .modal-header {
    background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
}

.modal-title-section {
    display: flex;
    align-items: center;
    gap: 15px;
}

.modal-icon {
    font-size: 2em;
}

.modal-title {
    margin: 0;
    font-size: 1.5em;
}

.modal-subtitle {
    margin: 5px 0 0 0;
    opacity: 0.9;
}

.close {
    position: absolute;
    top: 10px;
    right: 20px;
    color: white;
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
}

.close:hover {
    opacity: 0.7;
}

.modal-body {
    padding: 20px;
}

.form-actions {
    display: flex;
    justify-content: flex-end;
    gap: 10px;
    padding: 20px;
    border-top: 1px solid #e9ecef;
}
</style>

<script>
function approvePO(poId, poNumber) {
    document.getElementById('approval_po_id').value = poId;
    document.getElementById('approvalText').innerHTML = `Are you sure you want to approve Purchase Order <strong>${poNumber}</strong>?`;
    document.getElementById('approvalModal').style.display = 'block';
}

function rejectPO(poId, poNumber) {
    document.getElementById('rejection_po_id').value = poId;
    document.getElementById('rejectionText').innerHTML = `Are you sure you want to reject Purchase Order <strong>${poNumber}</strong>?`;
    document.getElementById('rejectionModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('approvalModal').style.display = 'none';
    document.getElementById('rejectionModal').style.display = 'none';
}

function applyFilters() {
    const searchValue = document.getElementById('search').value;
    const url = new URL(window.location);
    url.searchParams.set('search', searchValue);
    window.location.href = url.toString();
}

// Close modal when clicking outside
window.onclick = function(event) {
    const approvalModal = document.getElementById('approvalModal');
    const rejectionModal = document.getElementById('rejectionModal');
    if (event.target == approvalModal) {
        approvalModal.style.display = 'none';
    }
    if (event.target == rejectionModal) {
        rejectionModal.style.display = 'none';
    }
}
</script>
