<?php
session_start();

// Check permissions - only admins and accounts users can approve supplier invoices
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';
require_once 'accounting_functions.php';

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_status'])) {
        // Update invoice status
        $invoice_id = intval($_POST['invoice_id']);
        $new_status = $_POST['status'];

        $valid_statuses = ['pending_payment', 'cancelled'];
        if (!in_array($new_status, $valid_statuses)) {
            $error = "Invalid status.";
        } else {
            try {
                $stmt = $pdo->prepare("
                    UPDATE supplier_invoices
                    SET status = ?, updated_at = NOW()
                    WHERE id = ? AND status = 'pending_approval'
                ");
                $stmt->execute([$new_status, $invoice_id]);

                if ($stmt->rowCount() > 0) {
                    // Create journal entry for approval if status is pending_payment
                    if ($new_status === 'pending_payment') {
                        try {
                            createInvoiceApprovalEntry($pdo, $invoice_id, $_SESSION['user_id']);
                        } catch (Exception $e) {
                            $error = "Invoice status updated but accounting entry failed: " . $e->getMessage();
                        }
                    }

                    // Log activity
                    AccountantActivityLogger::logActivity(
                        $_SESSION['user_id'],
                        'supplier_invoice_status_updated',
                        "Updated supplier invoice ID $invoice_id status to $new_status"
                    );

                    $message = "Invoice status updated successfully!";
                } else {
                    $error = "Invoice not found or not in pending status.";
                }
            } catch (PDOException $e) {
                $error = "Error updating status: " . $e->getMessage();
            }
        }
    }
}

// Get all pending supplier invoices with PO and supplier info
try {
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';

    $query = "
        SELECT si.*, po.po_number, s.name AS supplier_name,
               u.username AS created_by_name,
               (
                   SELECT COUNT(*) FROM supplier_invoice_items sii WHERE sii.supplier_invoice_id = si.id
               ) as item_count
        FROM supplier_invoices si
        JOIN purchase_orders po ON si.purchase_order_id = po.id
        JOIN suppliers s ON po.supplier_id = s.id
        LEFT JOIN users u ON si.created_by = u.id
        WHERE si.status = 'pending_approval'
    ";

    $params = [];

    if (!empty($search)) {
        $query .= " AND (si.invoice_number LIKE ? OR po.po_number LIKE ? OR s.name LIKE ?)";
        $params[] = "%$search%";
        $params[] = "%$search%";
        $params[] = "%$search%";
    }

    $query .= " ORDER BY si.created_at DESC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $pending_invoices = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading pending invoices: " . $e->getMessage();
    $pending_invoices = [];
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .approve-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .approve-header {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .approve-content {
        display: grid;
        grid-template-columns: 1fr;
        gap: 30px;
    }

    .approve-section {
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .section-title {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
        font-size: 1.5em;
    }

    .filters-section {
        display: flex;
        gap: 20px;
        margin-bottom: 20px;
        align-items: center;
    }

    .filter-group {
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .filter-group label {
        font-weight: 600;
        color: #495057;
    }

    .filter-group input {
        padding: 8px 12px;
        border: 1px solid #ced4da;
        border-radius: 5px;
        font-size: 14px;
    }

    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        margin-right: 10px;
        transition: all 0.3s ease;
    }

    .btn-primary {
        background: #007bff;
        color: white;
    }

    .btn-primary:hover {
        background: #0056b3;
        transform: translateY(-2px);
    }

    .btn-success {
        background: #28a745;
        color: white;
    }

    .btn-success:hover {
        background: #1e7e34;
        transform: translateY(-2px);
    }

    .btn-danger {
        background: #dc3545;
        color: white;
    }

    .btn-danger:hover {
        background: #c82333;
        transform: translateY(-2px);
    }

    .message {
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }

    .success-message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .approve-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .approve-table th,
    .approve-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .approve-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #495057;
    }

    .invoice-number {
        font-weight: bold;
        color: #007bff;
    }

    .supplier-name {
        font-weight: 600;
        color: #28a745;
    }

    .amount {
        font-weight: 600;
        color: #007bff;
    }

    .actions-column {
        width: 200px;
    }

    .modal {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        z-index: 1000;
    }

    .modal-content {
        background: white;
        margin: 15% auto;
        padding: 30px;
        border-radius: 10px;
        width: 90%;
        max-width: 500px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.3);
    }

    .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
        padding-bottom: 10px;
        border-bottom: 1px solid #e9ecef;
    }

    .modal-title {
        margin: 0;
        color: #2c3e50;
    }

    .close {
        font-size: 24px;
        cursor: pointer;
        color: #6c757d;
    }

    @media (max-width: 768px) {
        .filters-section {
            flex-direction: column;
            align-items: stretch;
        }

        .approve-table {
            font-size: 0.9em;
        }

        .actions-column {
            width: 150px;
        }
    }
</style>

<div class="approve-container">
    <div class="approve-header">
        <h1>✅ Approve Supplier Invoices</h1>
        <p>Review and approve pending supplier invoices for payment</p>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="approve-content">
        <div class="approve-section">
            <div class="section-title">📋 Pending Invoices for Approval</div>

            <div class="filters-section">
                <div class="filter-group">
                    <label for="search">Search:</label>
                    <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($search ?? ''); ?>" placeholder="Invoice/PO number or supplier...">
                </div>
                <button class="btn btn-primary" onclick="applyFilters()">🔍 Filter</button>
            </div>

            <?php if (empty($pending_invoices)): ?>
                <p>No pending invoices to approve at this time.</p>
            <?php else: ?>
                <table class="approve-table">
                    <thead>
                        <tr>
                            <th>Invoice Number</th>
                            <th>PO Number</th>
                            <th>Supplier</th>
                            <th>Items</th>
                            <th>Total Amount</th>
                            <th>Due Date</th>
                            <th>Created</th>
                            <th class="actions-column">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($pending_invoices as $invoice): ?>
                            <tr>
                                <td>
                                    <div class="invoice-number"><?php echo htmlspecialchars($invoice['invoice_number']); ?></div>
                                </td>
                                <td>
                                    <div><?php echo htmlspecialchars($invoice['po_number']); ?></div>
                                </td>
                                <td>
                                    <div class="supplier-name"><?php echo htmlspecialchars($invoice['supplier_name']); ?></div>
                                </td>
                                <td><?php echo $invoice['item_count']; ?> items</td>
                                <td class="amount">GHC <?php echo number_format($invoice['total_amount'], 2); ?></td>
                                <td><?php echo date('M j, Y', strtotime($invoice['due_date'])); ?></td>
                                <td><?php echo date('M j, Y', strtotime($invoice['created_at'])); ?></td>
                                <td class="actions-column">
                                    <button class="btn btn-success btn-sm" onclick="updateStatus(<?php echo $invoice['id']; ?>, 'pending_payment')">✅ Approve</button>
                                    <button class="btn btn-danger btn-sm" onclick="updateStatus(<?php echo $invoice['id']; ?>, 'cancelled')">❌ Reject</button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Status Update Modal -->
<div id="statusModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Confirm Invoice Status Update</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <p>Are you sure you want to update this invoice status?</p>
        <form method="post">
            <input type="hidden" name="invoice_id" id="status_invoice_id">
            <input type="hidden" name="status" id="status_value">
            <button type="submit" name="update_status" class="btn btn-warning">Yes, Update Status</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<script>
function updateStatus(invoiceId, status) {
    document.getElementById('status_invoice_id').value = invoiceId;
    document.getElementById('status_value').value = status;
    document.getElementById('statusModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('statusModal').style.display = 'none';
}

function applyFilters() {
    const search = document.getElementById('search').value;

    let url = 'approve_invoices.php?';
    if (search) url += `search=${encodeURIComponent(search)}&`;

    window.location.href = url.slice(0, -1);
}

// Close modal when clicking outside
window.onclick = function(event) {
    if (event.target.className === 'modal') {
        event.target.style.display = 'none';
    }
}
</script>

<?php include '../includes/admin_footer.php'; ?>
