<?php
require_once '../config.php';
// require_once '../includes/access_control.php';

// requireFeatureAccess('accounts_management');

// Get dashboard data
try {
    // Current month expenses (using double-entry system)
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(jel.debit_amount), 0) as total_expenses
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE je.transaction_date >= DATE_FORMAT(CURRENT_DATE, '%Y-%m-01')
        AND je.transaction_date <= LAST_DAY(CURRENT_DATE)
        AND coa.account_type = 'expense'
        AND jel.debit_amount > 0
        AND je.status = 'posted'
    ");
    $stmt->execute();
    $current_month_expenses = $stmt->fetch()['total_expenses'] ?? 0;

    // Current month revenue (using double-entry system)
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(jel.credit_amount), 0) as total_revenue
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE je.transaction_date >= DATE_FORMAT(CURRENT_DATE, '%Y-%m-01')
        AND je.transaction_date <= LAST_DAY(CURRENT_DATE)
        AND coa.account_type = 'revenue'
        AND jel.credit_amount > 0
        AND je.status = 'posted'
    ");
    $stmt->execute();
    $current_month_revenue = $stmt->fetch()['total_revenue'] ?? 0;

    // Total expenses this year (using double-entry system)
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(jel.debit_amount), 0) as total_expenses
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE YEAR(je.transaction_date) = YEAR(CURRENT_DATE)
        AND coa.account_type = 'expense'
        AND jel.debit_amount > 0
        AND je.status = 'posted'
    ");
    $stmt->execute();
    $yearly_expenses = $stmt->fetch()['total_expenses'] ?? 0;

    // Total revenue this year (using double-entry system)
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(jel.credit_amount), 0) as total_revenue
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE YEAR(je.transaction_date) = YEAR(CURRENT_DATE)
        AND coa.account_type = 'revenue'
        AND jel.credit_amount > 0
        AND je.status = 'posted'
    ");
    $stmt->execute();
    $yearly_revenue = $stmt->fetch()['total_revenue'] ?? 0;

    // Recent expenses from journal entries (double-entry system)
    $stmt = $pdo->prepare("
        SELECT
            je.transaction_date as expense_date,
            je.description as title,
            jel.debit_amount as amount,
            coa.account_name as category_name,
            je.created_at,
            'posted' as status
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE je.source_type = 'expense'
        AND coa.account_type = 'expense'
        AND jel.debit_amount > 0
        AND je.status = 'posted'
        ORDER BY je.transaction_date DESC
        LIMIT 5
    ");
    $stmt->execute();
    $recent_expenses = $stmt->fetchAll();

    // Expense categories breakdown from journal entries (double-entry system)
    $stmt = $pdo->prepare("
        SELECT
            coa.account_name as category_name,
            SUM(jel.debit_amount) as total_amount,
            COUNT(DISTINCT je.id) as expense_count
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE YEAR(je.transaction_date) = YEAR(CURRENT_DATE)
        AND je.source_type = 'expense'
        AND coa.account_type = 'expense'
        AND jel.debit_amount > 0
        AND je.status = 'posted'
        GROUP BY coa.id, coa.account_name
        ORDER BY total_amount DESC
        LIMIT 10
    ");
    $stmt->execute();
    $expense_breakdown = $stmt->fetchAll();

    // Monthly trends (last 6 months) - using double-entry system
    $monthly_trends = [];
    for ($i = 5; $i >= 0; $i--) {
        $month = date('Y-m', strtotime("-$i months"));
        $month_name = date('M Y', strtotime("-$i months"));

        $stmt = $pdo->prepare("
            SELECT
                (SELECT COALESCE(SUM(jel.debit_amount), 0)
                 FROM journal_entries je
                 JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
                 JOIN chart_of_accounts coa ON jel.account_id = coa.id
                 WHERE DATE_FORMAT(je.transaction_date, '%Y-%m') = ?
                 AND coa.account_type = 'expense'
                 AND jel.debit_amount > 0
                 AND je.status = 'posted') as expenses,
                (SELECT COALESCE(SUM(jel.credit_amount), 0)
                 FROM journal_entries je
                 JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
                 JOIN chart_of_accounts coa ON jel.account_id = coa.id
                 WHERE DATE_FORMAT(je.transaction_date, '%Y-%m') = ?
                 AND coa.account_type = 'revenue'
                 AND jel.credit_amount > 0
                 AND je.status = 'posted') as revenue
        ");
        $stmt->execute([$month, $month]);
        $data = $stmt->fetch();

        $monthly_trends[] = [
            'month' => $month_name,
            'expenses' => $data['expenses'] ?? 0,
            'revenue' => $data['revenue'] ?? 0,
            'profit' => ($data['revenue'] ?? 0) - ($data['expenses'] ?? 0)
        ];
    }

    // Financial health indicators
    $current_ratio = $current_month_revenue > 0 ? $current_month_expenses / $current_month_revenue : 0;
    $yearly_profit = $yearly_revenue - $yearly_expenses;
    $profit_margin = $yearly_revenue > 0 ? ($yearly_profit / $yearly_revenue) * 100 : 0;

} catch (PDOException $e) {
    $error = "Error loading dashboard data: " . $e->getMessage();
    $current_month_expenses = $current_month_revenue = $yearly_expenses = $yearly_revenue = 0;
    $recent_expenses = $expense_breakdown = $monthly_trends = [];
    $current_ratio = $yearly_profit = $profit_margin = 0;
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .dashboard-content {
        padding: 20px;
    }

    .dashboard-header {
        background: white;
        padding: 30px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .dashboard-header h1 {
        color: #2c3e50;
        margin: 0 0 10px 0;
        font-size: 2.5em;
    }

    .dashboard-header p {
        color: #7f8c8d;
        margin: 0;
        font-size: 1.1em;
    }

    .dashboard-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }

    .dashboard-card {
        background: white;
        padding: 25px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        transition: transform 0.2s;
    }

    .dashboard-card:hover {
        transform: translateY(-5px);
    }

    .card-header {
        display: flex;
        align-items: center;
        margin-bottom: 20px;
    }

    .card-icon {
        font-size: 2.5em;
        margin-right: 15px;
    }

    .card-title {
        margin: 0;
        color: #2c3e50;
        font-size: 1.3em;
    }

    .card-value {
        font-size: 2em;
        font-weight: 600;
        margin: 10px 0;
    }

    .card-subtitle {
        color: #7f8c8d;
        font-size: 0.9em;
    }

    .revenue-card .card-icon { color: #27ae60; }
    .revenue-card .card-value { color: #27ae60; }
    .expense-card .card-icon { color: #e74c3c; }
    .expense-card .card-value { color: #e74c3c; }
    .profit-card .card-icon { color: #3498db; }
    .profit-card .card-value { color: #3498db; }
    .ratio-card .card-icon { color: #f39c12; }
    .ratio-card .card-value { color: #f39c12; }

    .quick-actions {
        background: white;
        padding: 25px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        margin-bottom: 30px;
    }

    .quick-actions h2 {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
    }

    .actions-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 15px;
    }

    .action-btn {
        display: flex;
        align-items: center;
        padding: 15px 20px;
        background: #f8f9fa;
        border: 1px solid #e9ecef;
        border-radius: 8px;
        text-decoration: none;
        color: #495057;
        transition: all 0.2s;
    }

    .action-btn:hover {
        background: #2E7D32;
        color: white;
        border-color: #2E7D32;
        transform: translateY(-2px);
    }

    .action-btn .icon {
        font-size: 1.5em;
        margin-right: 10px;
    }

    .charts-section {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
        gap: 30px;
        margin-bottom: 30px;
    }

    .chart-container {
        background: white;
        padding: 20px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .chart-container h3 {
        color: #2c3e50;
        margin: 0 0 20px 0;
        text-align: center;
    }

    .recent-activity {
        background: white;
        padding: 25px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .recent-activity h2 {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
    }

    .activity-item {
        display: flex;
        align-items: center;
        padding: 15px 0;
        border-bottom: 1px solid #f0f0f0;
    }

    .activity-item:last-child {
        border-bottom: none;
    }

    .activity-icon {
        font-size: 1.5em;
        margin-right: 15px;
        width: 30px;
        text-align: center;
    }

    .activity-content {
        flex: 1;
    }

    .activity-title {
        font-weight: 600;
        color: #2c3e50;
        margin: 0 0 5px 0;
    }

    .activity-meta {
        font-size: 0.9em;
        color: #7f8c8d;
    }

    .expense-item .activity-icon { color: #e74c3c; }
    .revenue-item .activity-icon { color: #27ae60; }

    .financial-health {
        background: white;
        padding: 25px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .financial-health h2 {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
    }

    .health-indicators {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
    }

    .health-indicator {
        text-align: center;
        padding: 15px;
        border-radius: 8px;
        background: #f8f9fa;
    }

    .health-indicator h4 {
        margin: 0 0 10px 0;
        color: #495057;
    }

    .health-value {
        font-size: 1.5em;
        font-weight: 600;
        margin: 0 0 5px 0;
    }

    .health-status {
        font-size: 0.9em;
        padding: 5px 10px;
        border-radius: 12px;
        display: inline-block;
    }

    .status-excellent { background: #d4edda; color: #155724; }
    .status-good { background: #d1ecf1; color: #0c5460; }
    .status-fair { background: #fff3cd; color: #856404; }
    .status-poor { background: #f8d7da; color: #721c24; }

    .table-container {
        overflow-x: auto;
        margin-top: 10px;
    }

    .recent-entries-table {
        width: 100%;
        border-collapse: collapse;
        background: white;
    }

    .recent-entries-table th {
        background: #f8f9fa;
        color: #2c3e50;
        padding: 12px;
        text-align: left;
        border-bottom: 2px solid #dee2e6;
        font-weight: 600;
    }

    .recent-entries-table td {
        padding: 12px;
        border-bottom: 1px solid #dee2e6;
    }

    .recent-entries-table tr:hover {
        background: #f8f9fa;
    }

    .status-badge {
        padding: 4px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: bold;
        text-transform: uppercase;
    }

    .status-posted { background: #d4edda; color: #155724; }
    .status-draft { background: #fff3cd; color: #856404; }
    .status-pending { background: #d1ecf1; color: #0c5460; }

    @media (max-width: 768px) {
        .dashboard-grid {
            grid-template-columns: 1fr;
        }

        .charts-section {
            grid-template-columns: 1fr;
        }

        .actions-grid {
            grid-template-columns: 1fr;
        }

        .health-indicators {
            grid-template-columns: 1fr;
        }
    }
</style>

<div class="dashboard-content">
        <div class="dashboard-header">
            <h1>📊 Accounting Dashboard</h1>
            <p>Financial overview and management for your school</p>
        </div>

        <!-- Key Metrics -->
        <div class="dashboard-grid">
            <div class="dashboard-card revenue-card">
                <div class="card-header">
                    <div class="card-icon">💰</div>
                    <h3 class="card-title">Monthly Revenue</h3>
                </div>
                <div class="card-value">GHC <?php echo number_format($current_month_revenue, 2); ?></div>
                <div class="card-subtitle">Current month income</div>
            </div>

            <div class="dashboard-card expense-card">
                <div class="card-header">
                    <div class="card-icon">💸</div>
                    <h3 class="card-title">Monthly Expenses</h3>
                </div>
                <div class="card-value">GHC <?php echo number_format($current_month_expenses, 2); ?></div>
                <div class="card-subtitle">Current month spending</div>
            </div>

            <div class="dashboard-card profit-card">
                <div class="card-header">
                    <div class="card-icon">📈</div>
                    <h3 class="card-title">Monthly Profit</h3>
                </div>
                <div class="card-value">GHC <?php echo number_format($current_month_revenue - $current_month_expenses, 2); ?></div>
                <div class="card-subtitle">Revenue - Expenses</div>
            </div>

            <div class="dashboard-card ratio-card">
                <div class="card-header">
                    <div class="card-icon">⚖️</div>
                    <h3 class="card-title">Expense Ratio</h3>
                </div>
                <div class="card-value"><?php echo $current_month_revenue > 0 ? round(($current_month_expenses / $current_month_revenue) * 100, 1) : 0; ?>%</div>
                <div class="card-subtitle">Expenses / Revenue</div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="quick-actions">
            <h2>🚀 Quick Actions</h2>
            <div class="actions-grid">
                <a href="record_expense.php" class="action-btn">
                    <span class="icon">➕</span>
                    Record New Expense
                </a>
                <a href="expense_list.php" class="action-btn">
                    <span class="icon">📋</span>
                    Manage Expenses
                </a>
                <a href="balance_sheet_report.php" class="action-btn">
                    <span class="icon">📊</span>
                    Balance Sheet
                </a>
                <a href="profit_loss_report.php" class="action-btn">
                    <span class="icon">📈</span>
                    Profit & Loss
                </a>
                <a href="cash_flow_statement.php" class="action-btn">
                    <span class="icon">💹</span>
                    Cash Flow
                </a>
                <a href="budget_vs_actual_report.php" class="action-btn">
                    <span class="icon">📋</span>
                    Budget vs Actual
                </a>
            </div>
        </div>

        <!-- Charts and Recent Activity -->
        <div style="display: grid; grid-template-columns: 2fr 1fr; gap: 30px;">
            <!-- Financial Trends Chart -->
            <div class="chart-container">
                <h3>📈 Financial Trends (Last 6 Months)</h3>
                <canvas id="trendsChart" width="400" height="200"></canvas>
            </div>

            <!-- Recent Activity -->
            <div class="recent-activity">
                <h2>🕒 Recent Expenses</h2>
                <?php if (empty($recent_expenses)): ?>
                    <p style="color: #7f8c8d; font-style: italic; text-align: center; padding: 20px;">
                        No recent expenses
                    </p>
                <?php else: ?>
                    <?php foreach ($recent_expenses as $expense): ?>
                        <div class="activity-item expense-item">
                            <div class="activity-icon">💸</div>
                            <div class="activity-content">
                                <h4 class="activity-title"><?php echo htmlspecialchars($expense['title']); ?></h4>
                                <div class="activity-meta">
                                    GHC <?php echo number_format($expense['amount'], 2); ?> •
                                    <?php echo htmlspecialchars($expense['category_name'] ?? 'Uncategorized'); ?> •
                                    <?php echo date('M d, Y', strtotime($expense['expense_date'])); ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Recent Journal Entries -->
        <div class="recent-activity">
            <h2>📝 Recent Journal Entries</h2>
            <div class="table-container">
                <table class="recent-entries-table">
                    <thead>
                        <tr>
                            <th>Entry #</th>
                            <th>Date</th>
                            <th>Description</th>
                            <th>Source</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $stmt = $pdo->prepare("
                            SELECT je.entry_number, je.transaction_date, je.description, je.source_type, je.status
                            FROM journal_entries je
                            WHERE je.status = 'posted'
                            ORDER BY je.id DESC
                            LIMIT 10
                        ");
                        $stmt->execute();
                        $recent_entries = $stmt->fetchAll();

                        if (empty($recent_entries)): ?>
                            <tr>
                                <td colspan="5" style="text-align: center; color: #7f8c8d;">No journal entries found</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($recent_entries as $entry): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($entry['entry_number']); ?></td>
                                    <td><?php echo date('M d, Y', strtotime($entry['transaction_date'])); ?></td>
                                    <td><?php echo htmlspecialchars(substr($entry['description'], 0, 50)) . (strlen($entry['description']) > 50 ? '...' : ''); ?></td>
                                    <td><?php echo ucfirst(htmlspecialchars($entry['source_type'])); ?></td>
                                    <td>
                                        <span class="status-badge status-<?php echo $entry['status']; ?>">
                                            <?php echo ucfirst($entry['status']); ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Financial Health Indicators -->
        <div class="financial-health">
            <h2>💡 Financial Health</h2>
            <div class="health-indicators">
                <div class="health-indicator">
                    <h4>Current Ratio</h4>
                    <div class="health-value"><?php echo round($current_month_revenue / max($current_month_expenses, 1), 2); ?></div>
                    <div class="health-status status-<?php
                        $ratio = $current_month_revenue / max($current_month_expenses, 1);
                        echo $ratio >= 2 ? 'excellent' : ($ratio >= 1.5 ? 'good' : ($ratio >= 1 ? 'fair' : 'poor'));
                    ?>">
                        <?php
                        $ratio = $current_month_revenue / max($current_month_expenses, 1);
                        if ($ratio >= 2) echo 'Excellent';
                        elseif ($ratio >= 1.5) echo 'Good';
                        elseif ($ratio >= 1) echo 'Fair';
                        else echo 'Poor';
                        ?>
                    </div>
                </div>

                <div class="health-indicator">
                    <h4>Yearly Profit</h4>
                    <div class="health-value">GHC <?php echo number_format($yearly_profit, 2); ?></div>
                    <div class="health-status status-<?php echo $yearly_profit >= 0 ? 'excellent' : 'poor'; ?>">
                        <?php echo $yearly_profit >= 0 ? 'Profitable' : 'Loss'; ?>
                    </div>
                </div>

                <div class="health-indicator">
                    <h4>Profit Margin</h4>
                    <div class="health-value"><?php echo round($profit_margin, 1); ?>%</div>
                    <div class="health-status status-<?php
                        echo $profit_margin >= 20 ? 'excellent' : ($profit_margin >= 10 ? 'good' : ($profit_margin >= 0 ? 'fair' : 'poor'));
                    ?>">
                        <?php
                        if ($profit_margin >= 20) echo 'Excellent';
                        elseif ($profit_margin >= 10) echo 'Good';
                        elseif ($profit_margin >= 0) echo 'Fair';
                        else echo 'Poor';
                        ?>
                    </div>
                </div>

                <div class="health-indicator">
                    <h4>Expense Control</h4>
                    <div class="health-value"><?php echo $yearly_revenue > 0 ? round(($yearly_expenses / $yearly_revenue) * 100, 1) : 0; ?>%</div>
                    <div class="health-status status-<?php
                        $expense_ratio = $yearly_revenue > 0 ? ($yearly_expenses / $yearly_revenue) * 100 : 0;
                        echo $expense_ratio <= 60 ? 'excellent' : ($expense_ratio <= 80 ? 'good' : ($expense_ratio <= 100 ? 'fair' : 'poor'));
                    ?>">
                        <?php
                        $expense_ratio = $yearly_revenue > 0 ? ($yearly_expenses / $yearly_revenue) * 100 : 0;
                        if ($expense_ratio <= 60) echo 'Excellent';
                        elseif ($expense_ratio <= 80) echo 'Good';
                        elseif ($expense_ratio <= 100) echo 'Fair';
                        else echo 'Poor';
                        ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Financial Trends Chart
            const trendsCtx = document.getElementById('trendsChart').getContext('2d');
            const trendLabels = <?php echo json_encode(array_column($monthly_trends, 'month')); ?>;
            const revenueData = <?php echo json_encode(array_column($monthly_trends, 'revenue')); ?>;
            const expenseData = <?php echo json_encode(array_column($monthly_trends, 'expenses')); ?>;
            const profitData = <?php echo json_encode(array_column($monthly_trends, 'profit')); ?>;

            new Chart(trendsCtx, {
                type: 'line',
                data: {
                    labels: trendLabels,
                    datasets: [
                        {
                            label: 'Revenue',
                            data: revenueData,
                            borderColor: 'rgba(46, 204, 113, 1)',
                            backgroundColor: 'rgba(46, 204, 113, 0.1)',
                            tension: 0.4
                        },
                        {
                            label: 'Expenses',
                            data: expenseData,
                            borderColor: 'rgba(231, 76, 60, 1)',
                            backgroundColor: 'rgba(231, 76, 60, 0.1)',
                            tension: 0.4
                        },
                        {
                            label: 'Profit',
                            data: profitData,
                            borderColor: 'rgba(52, 152, 219, 1)',
                            backgroundColor: 'rgba(52, 152, 219, 0.1)',
                            tension: 0.4
                        }
                    ]
                },
                options: {
                    responsive: true,
                    scales: {
                        y: {
                            ticks: {
                                callback: function(value) {
                                    return 'GHC ' + value.toLocaleString();
                                }
                            }
                        }
                    },
                    plugins: {
                        legend: {
                            position: 'top'
                        }
                    }
                }
            });
        });
    </script>

<?php include '../includes/admin_footer.php'; ?>
